
package com.gridnine.xtrip.server.gds.tandc.wsmodel;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/** 
 * The OrderReshop transaction set passes new shopping requests to an airline to replace existing specified order or order items in an Order or for new shopping requests to add to an existing Order. Airline responds with product offers within the context of the existing Order.

 There is no change to the Order or Order Items, in response to a Reshop request - typically, the Reshop response will be followed by an OrderChangeRQ or OrderCancelRQ transaction, to commit the re-shop offer.

 In the response, the Airline can provide exact instructions as to what actions can finally be executed, as a solution to the initially requested action.
 * 
 * Schema fragment(s) for this class:
 * <pre>
 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="OrderReshopRQ">
 *   &lt;xs:complexType>
 *     &lt;xs:sequence>
 *       &lt;xs:element ref="ns:PointOfSale" minOccurs="0"/>
 *       &lt;xs:element ref="ns:Document"/>
 *       &lt;xs:element ref="ns:Party"/>
 *       &lt;xs:element ref="ns:ReshopParameters" minOccurs="0"/>
 *       &lt;xs:element name="Query">
 *         &lt;xs:complexType>
 *           &lt;xs:sequence>
 *             &lt;xs:element type="xs:string" name="OrderID"/>
 *             &lt;xs:element ref="ns:BookingReferences" minOccurs="0"/>
 *             &lt;xs:element type="xs:string" name="ActionContext" minOccurs="0"/>
 *             &lt;xs:choice>
 *               &lt;xs:element name="Reshop">
 *                 &lt;xs:complexType>
 *                   &lt;xs:choice>
 *                     &lt;xs:element name="OrderServicing">
 *                       &lt;!-- Reference to inner class OrderServicing -->
 *                     &lt;/xs:element>
 *                     &lt;xs:element name="NameChange">
 *                       &lt;xs:complexType>
 *                         &lt;xs:sequence>
 *                           &lt;xs:element type="xs:string" name="PassengerID"/>
 *                           &lt;xs:element type="xs:string" name="NameTitle" minOccurs="0"/>
 *                           &lt;xs:element type="xs:string" name="GivenName" minOccurs="0" maxOccurs="5"/>
 *                           &lt;xs:element type="xs:string" name="MiddleName" minOccurs="0" maxOccurs="3"/>
 *                           &lt;xs:element type="xs:string" name="Surname" minOccurs="0"/>
 *                           &lt;xs:element type="xs:string" name="SurnameSuffix" minOccurs="0"/>
 *                         &lt;/xs:sequence>
 *                       &lt;/xs:complexType>
 *                     &lt;/xs:element>
 *                   &lt;/xs:choice>
 *                 &lt;/xs:complexType>
 *               &lt;/xs:element>
 *               &lt;xs:element name="Reprice">
 *                 &lt;!-- Reference to inner class Reprice -->
 *               &lt;/xs:element>
 *             &lt;/xs:choice>
 *           &lt;/xs:sequence>
 *         &lt;/xs:complexType>
 *       &lt;/xs:element>
 *       &lt;xs:element name="DataLists" minOccurs="0">
 *         &lt;!-- Reference to inner class DataLists -->
 *       &lt;/xs:element>
 *     &lt;/xs:sequence>
 *     &lt;xs:attributeGroup ref="ns:IATA_PayloadStdAttributes"/>
 *   &lt;/xs:complexType>
 * &lt;/xs:element>
 * </pre>
 */
public class OrderReshopRQ implements Serializable
{
    private PointOfSaleType pointOfSale;
    private MsgDocumentType document;
    private MsgPartiesType party;
    private ItinReshopParamsType reshopParameters;
    private String queryOrderID;
    private BookingReferences queryBookingReferences;
    private String queryActionContext;
    private int queryChoiceSelect = -1;
    private static final int QUERY_RESHOP_CHOICE = 0;
    private static final int REPRICE_CHOICE = 1;
    private int queryReshopSelect = -1;
    private static final int ORDER_SERVICING_CHOICE = 0;
    private static final int QUERY_NAME_CHANGE_CHOICE = 1;
    private OrderServicing orderServicing;
    private String queryPassengerID;
    private String queryNameTitle;
    private List<String> queryGivenNameList = new ArrayList<String>();
    private List<String> queryMiddleNameList = new ArrayList<String>();
    private String querySurname;
    private String querySurnameSuffix;
    private Reprice reprice;
    private DataLists dataLists;
    private IATAPayloadStdAttributes IATAPayloadStdAttributes;

    /** 
     * Get the 'PointOfSale' element value.
     * 
     * @return value
     */
    public PointOfSaleType getPointOfSale() {
        return pointOfSale;
    }

    /** 
     * Set the 'PointOfSale' element value.
     * 
     * @param pointOfSale
     */
    public void setPointOfSale(PointOfSaleType pointOfSale) {
        this.pointOfSale = pointOfSale;
    }

    /** 
     * Get the 'Document' element value.
     * 
     * @return value
     */
    public MsgDocumentType getDocument() {
        return document;
    }

    /** 
     * Set the 'Document' element value.
     * 
     * @param document
     */
    public void setDocument(MsgDocumentType document) {
        this.document = document;
    }

    /** 
     * Get the 'Party' element value.
     * 
     * @return value
     */
    public MsgPartiesType getParty() {
        return party;
    }

    /** 
     * Set the 'Party' element value.
     * 
     * @param party
     */
    public void setParty(MsgPartiesType party) {
        this.party = party;
    }

    /** 
     * Get the 'ReshopParameters' element value.
     * 
     * @return value
     */
    public ItinReshopParamsType getReshopParameters() {
        return reshopParameters;
    }

    /** 
     * Set the 'ReshopParameters' element value.
     * 
     * @param reshopParameters
     */
    public void setReshopParameters(ItinReshopParamsType reshopParameters) {
        this.reshopParameters = reshopParameters;
    }

    /** 
     * Get the 'OrderID' element value. Reference to existing Order to which the Seller wants to add, udpate, delete an OrderItem.
     * 
     * @return value
     */
    public String getQueryOrderID() {
        return queryOrderID;
    }

    /** 
     * Set the 'OrderID' element value. Reference to existing Order to which the Seller wants to add, udpate, delete an OrderItem.
     * 
     * @param queryOrderID
     */
    public void setQueryOrderID(String queryOrderID) {
        this.queryOrderID = queryOrderID;
    }

    /** 
     * Get the 'BookingReferences' element value. Existing Booking Reference (e.g. PNR Record Locator). Example: R1A3SK
     * 
     * @return value
     */
    public BookingReferences getQueryBookingReferences() {
        return queryBookingReferences;
    }

    /** 
     * Set the 'BookingReferences' element value. Existing Booking Reference (e.g. PNR Record Locator). Example: R1A3SK
     * 
     * @param queryBookingReferences
     */
    public void setQueryBookingReferences(
            BookingReferences queryBookingReferences) {
        this.queryBookingReferences = queryBookingReferences;
    }

    /** 
     * Get the 'ActionContext' element value. Reference to PADIS codeset REA. When used, provides context for the change being requested.
     * 
     * @return value
     */
    public String getQueryActionContext() {
        return queryActionContext;
    }

    /** 
     * Set the 'ActionContext' element value. Reference to PADIS codeset REA. When used, provides context for the change being requested.
     * 
     * @param queryActionContext
     */
    public void setQueryActionContext(String queryActionContext) {
        this.queryActionContext = queryActionContext;
    }

    private void setQueryChoiceSelect(int choice) {
        if (queryChoiceSelect == -1) {
            queryChoiceSelect = choice;
        } else if (queryChoiceSelect != choice) {
            throw new IllegalStateException(
                    "Need to call clearQueryChoiceSelect() before changing existing choice");
        }
    }

    /** 
     * Clear the choice selection.
     */
    public void clearQueryChoiceSelect() {
        queryChoiceSelect = -1;
    }

    /** 
     * Check if QueryReshop is current selection for choice.
     * 
     * @return <code>true</code> if selection, <code>false</code> if not
     */
    public boolean ifQueryReshop() {
        return queryChoiceSelect == QUERY_RESHOP_CHOICE;
    }

    private void setQueryReshopSelect(int choice) {
        setQueryChoiceSelect(QUERY_RESHOP_CHOICE);
        if (queryReshopSelect == -1) {
            queryReshopSelect = choice;
        } else if (queryReshopSelect != choice) {
            throw new IllegalStateException(
                    "Need to call clearQueryReshopSelect() before changing existing choice");
        }
    }

    /** 
     * Clear the choice selection.
     */
    public void clearQueryReshopSelect() {
        queryReshopSelect = -1;
    }

    /** 
     * Check if OrderServicing is current selection for choice.
     * 
     * @return <code>true</code> if selection, <code>false</code> if not
     */
    public boolean ifOrderServicing() {
        return queryReshopSelect == ORDER_SERVICING_CHOICE;
    }

    /** 
     * Get the 'OrderServicing' element value. Function used to reshop items within an order (e.g. adding ancillaries, cancelling order items, replacing order items).
     * 
     * @return value
     */
    public OrderServicing getOrderServicing() {
        return orderServicing;
    }

    /** 
     * Set the 'OrderServicing' element value. Function used to reshop items within an order (e.g. adding ancillaries, cancelling order items, replacing order items).
     * 
     * @param orderServicing
     */
    public void setOrderServicing(OrderServicing orderServicing) {
        setQueryReshopSelect(ORDER_SERVICING_CHOICE);
        this.orderServicing = orderServicing;
    }

    /** 
     * Check if QueryNameChange is current selection for choice.
     * 
     * @return <code>true</code> if selection, <code>false</code> if not
     */
    public boolean ifQueryNameChange() {
        return queryReshopSelect == QUERY_NAME_CHANGE_CHOICE;
    }

    /** 
     * Get the 'PassengerID' element value. Identifier of the Passenger within the Order in the Order Management System. This is the PassengerID that is assigned at the time of Order creation by the Airline.
     * 
     * @return value
     */
    public String getQueryPassengerID() {
        return queryPassengerID;
    }

    /** 
     * Set the 'PassengerID' element value. Identifier of the Passenger within the Order in the Order Management System. This is the PassengerID that is assigned at the time of Order creation by the Airline.
     * 
     * @param queryPassengerID
     */
    public void setQueryPassengerID(String queryPassengerID) {
        setQueryReshopSelect(QUERY_NAME_CHANGE_CHOICE);
        this.queryPassengerID = queryPassengerID;
    }

    /** 
     * Get the 'NameTitle' element value. Individual's name title. Examples: MR, MRS, DRn.
     * 
     * @return value
     */
    public String getQueryNameTitle() {
        return queryNameTitle;
    }

    /** 
     * Set the 'NameTitle' element value. Individual's name title. Examples: MR, MRS, DRn.
     * 
     * @param queryNameTitle
     */
    public void setQueryNameTitle(String queryNameTitle) {
        setQueryReshopSelect(QUERY_NAME_CHANGE_CHOICE);
        this.queryNameTitle = queryNameTitle;
    }

    /** 
     * Get the list of 'GivenName' element items. Individual's first or given name. Example: JOHN.
     * 
     * @return list
     */
    public List<String> getQueryGivenNameList() {
        return queryGivenNameList;
    }

    /** 
     * Set the list of 'GivenName' element items. Individual's first or given name. Example: JOHN.
     * 
     * @param list
     */
    public void setQueryGivenNameList(List<String> list) {
        setQueryReshopSelect(QUERY_NAME_CHANGE_CHOICE);
        queryGivenNameList = list;
    }

    /** 
     * Get the list of 'MiddleName' element items. Individual's middle name or initial.  Examples: G, GEORGE
     * 
     * @return list
     */
    public List<String> getQueryMiddleNameList() {
        return queryMiddleNameList;
    }

    /** 
     * Set the list of 'MiddleName' element items. Individual's middle name or initial.  Examples: G, GEORGE
     * 
     * @param list
     */
    public void setQueryMiddleNameList(List<String> list) {
        setQueryReshopSelect(QUERY_NAME_CHANGE_CHOICE);
        queryMiddleNameList = list;
    }

    /** 
     * Get the 'Surname' element value. Individual's family name, last name. Example: SMITH.
     * 
     * @return value
     */
    public String getQuerySurname() {
        return querySurname;
    }

    /** 
     * Set the 'Surname' element value. Individual's family name, last name. Example: SMITH.
     * 
     * @param querySurname
     */
    public void setQuerySurname(String querySurname) {
        setQueryReshopSelect(QUERY_NAME_CHANGE_CHOICE);
        this.querySurname = querySurname;
    }

    /** 
     * Get the 'SurnameSuffix' element value. Individual's surname suffixes and letters. Examples: Jr., Sr., III, Ret., Esq.
     * 
     * @return value
     */
    public String getQuerySurnameSuffix() {
        return querySurnameSuffix;
    }

    /** 
     * Set the 'SurnameSuffix' element value. Individual's surname suffixes and letters. Examples: Jr., Sr., III, Ret., Esq.
     * 
     * @param querySurnameSuffix
     */
    public void setQuerySurnameSuffix(String querySurnameSuffix) {
        setQueryReshopSelect(QUERY_NAME_CHANGE_CHOICE);
        this.querySurnameSuffix = querySurnameSuffix;
    }

    /** 
     * Check if Reprice is current selection for choice.
     * 
     * @return <code>true</code> if selection, <code>false</code> if not
     */
    public boolean ifReprice() {
        return queryChoiceSelect == REPRICE_CHOICE;
    }

    /** 
     * Get the 'Reprice' element value.
     * 
     * @return value
     */
    public Reprice getReprice() {
        return reprice;
    }

    /** 
     * Set the 'Reprice' element value.
     * 
     * @param reprice
     */
    public void setReprice(Reprice reprice) {
        setQueryChoiceSelect(REPRICE_CHOICE);
        this.reprice = reprice;
    }

    /** 
     * Get the 'DataLists' element value.
     * 
     * @return value
     */
    public DataLists getDataLists() {
        return dataLists;
    }

    /** 
     * Set the 'DataLists' element value.
     * 
     * @param dataLists
     */
    public void setDataLists(DataLists dataLists) {
        this.dataLists = dataLists;
    }

    /** 
     * Get the 'IATA_PayloadStdAttributes' attributeGroup value.
     * 
     * @return value
     */
    public IATAPayloadStdAttributes getIATAPayloadStdAttributes() {
        return IATAPayloadStdAttributes;
    }

    /** 
     * Set the 'IATA_PayloadStdAttributes' attributeGroup value.
     * 
     * @param IATAPayloadStdAttributes
     */
    public void setIATAPayloadStdAttributes(
            IATAPayloadStdAttributes IATAPayloadStdAttributes) {
        this.IATAPayloadStdAttributes = IATAPayloadStdAttributes;
    }
    /** 
     * Functions to allow a Seller to request additions, replacement or deletions of OrderItems in this specific Order.
     * 
     * Schema fragment(s) for this class:
     * <pre>
     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="OrderServicing">
     *   &lt;xs:complexType>
     *     &lt;xs:sequence>
     *       &lt;xs:element name="Add" minOccurs="0">
     *         &lt;!-- Reference to inner class Add -->
     *       &lt;/xs:element>
     *       &lt;xs:element name="Delete" minOccurs="0">
     *         &lt;!-- Reference to inner class Delete -->
     *       &lt;/xs:element>
     *     &lt;/xs:sequence>
     *   &lt;/xs:complexType>
     * &lt;/xs:element>
     * </pre>
     */
    public static class OrderServicing implements Serializable
    {
        private Add add;
        private Delete delete;

        /** 
         * Get the 'Add' element value.
         * 
         * @return value
         */
        public Add getAdd() {
            return add;
        }

        /** 
         * Set the 'Add' element value.
         * 
         * @param add
         */
        public void setAdd(Add add) {
            this.add = add;
        }

        /** 
         * Get the 'Delete' element value.
         * 
         * @return value
         */
        public Delete getDelete() {
            return delete;
        }

        /** 
         * Set the 'Delete' element value.
         * 
         * @param delete
         */
        public void setDelete(Delete delete) {
            this.delete = delete;
        }
        /** 
         * Action to request new Offer Items for this specific Order.

        If used in conjunction with a “Delete” function, the intention is to replace the deleted item with whatever has been requested in this Add function’s shopping query.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Add" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element name="FlightQuery" minOccurs="0">
         *         &lt;!-- Reference to inner class FlightQuery -->
         *       &lt;/xs:element>
         *       &lt;xs:element name="Qualifier" minOccurs="0" maxOccurs="unbounded">
         *         &lt;!-- Reference to inner class Qualifier -->
         *       &lt;/xs:element>
         *       &lt;xs:element name="Preference" minOccurs="0">
         *         &lt;!-- Reference to inner class Preference -->
         *       &lt;/xs:element>
         *       &lt;xs:element name="Passengers" minOccurs="0">
         *         &lt;xs:complexType>
         *           &lt;xs:sequence>
         *             &lt;xs:element ref="ns:PassengerReferences" minOccurs="0"/>
         *           &lt;/xs:sequence>
         *         &lt;/xs:complexType>
         *       &lt;/xs:element>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class Add implements Serializable
        {
            private FlightQuery flightQuery;
            private List<Qualifier> qualifierList = new ArrayList<Qualifier>();
            private Preference preference;
            private PassengerReferences passengers;

            /** 
             * Get the 'FlightQuery' element value.
             * 
             * @return value
             */
            public FlightQuery getFlightQuery() {
                return flightQuery;
            }

            /** 
             * Set the 'FlightQuery' element value.
             * 
             * @param flightQuery
             */
            public void setFlightQuery(FlightQuery flightQuery) {
                this.flightQuery = flightQuery;
            }

            /** 
             * Get the list of 'Qualifier' element items.
             * 
             * @return list
             */
            public List<Qualifier> getQualifierList() {
                return qualifierList;
            }

            /** 
             * Set the list of 'Qualifier' element items.
             * 
             * @param list
             */
            public void setQualifierList(List<Qualifier> list) {
                qualifierList = list;
            }

            /** 
             * Get the 'Preference' element value.
             * 
             * @return value
             */
            public Preference getPreference() {
                return preference;
            }

            /** 
             * Set the 'Preference' element value.
             * 
             * @param preference
             */
            public void setPreference(Preference preference) {
                this.preference = preference;
            }

            /** 
             * Get the 'PassengerReferences' element value. Reference to passenger to be added to existing order.
             * 
             * @return value
             */
            public PassengerReferences getPassengers() {
                return passengers;
            }

            /** 
             * Set the 'PassengerReferences' element value. Reference to passenger to be added to existing order.
             * 
             * @param passengers
             */
            public void setPassengers(PassengerReferences passengers) {
                this.passengers = passengers;
            }
            /** 
             * [Re-using the query structure from AirShoppingRQ]

            Core Air Shopping information by supported Air Shopping Query type.

            Shopping Query Input may be specified as:
            1) Prior Shopping Session Basket
            2) Attribute Shopping Query
            3) Affinity Shopping Query
             * 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="FlightQuery" minOccurs="0">
             *   &lt;xs:complexType>
             *     &lt;xs:choice>
             *       &lt;xs:element name="Affinity">
             *         &lt;xs:complexType>
             *           &lt;xs:sequence>
             *             &lt;xs:element name="OriginDestination" maxOccurs="unbounded">
             *               &lt;!-- Reference to inner class OriginDestination -->
             *             &lt;/xs:element>
             *             &lt;xs:element ref="ns:StayPeriodDateRange" minOccurs="0"/>
             *             &lt;xs:element ref="ns:StayPeriodSeason" minOccurs="0"/>
             *             &lt;xs:element ref="ns:TripDistance" minOccurs="0"/>
             *             &lt;xs:element ref="ns:TripTime" minOccurs="0"/>
             *             &lt;xs:element name="Budget" minOccurs="0">
             *               &lt;!-- Reference to inner class Budget -->
             *             &lt;/xs:element>
             *             &lt;xs:element name="KeywordPreferences" minOccurs="0">
             *               &lt;!-- Reference to inner class KeywordPreferences -->
             *             &lt;/xs:element>
             *             &lt;xs:element ref="ns:ShoppingResponseID" minOccurs="0"/>
             *           &lt;/xs:sequence>
             *         &lt;/xs:complexType>
             *       &lt;/xs:element>
             *       &lt;xs:element name="OriginDestinations">
             *         &lt;xs:complexType>
             *           &lt;xs:sequence>
             *             &lt;xs:element name="OriginDestination" maxOccurs="unbounded">
             *               &lt;!-- Reference to inner class OriginDestination1 -->
             *             &lt;/xs:element>
             *           &lt;/xs:sequence>
             *         &lt;/xs:complexType>
             *       &lt;/xs:element>
             *       &lt;xs:element name="FlightSpecific">
             *         &lt;xs:complexType>
             *           &lt;xs:choice>
             *             &lt;xs:element name="FlightSegment" maxOccurs="unbounded">
             *               &lt;!-- Reference to inner class FlightSegment -->
             *             &lt;/xs:element>
             *             &lt;xs:element name="Flight" maxOccurs="unbounded">
             *               &lt;!-- Reference to inner class Flight -->
             *             &lt;/xs:element>
             *             &lt;xs:element ref="ns:OriginDestination" maxOccurs="unbounded"/>
             *           &lt;/xs:choice>
             *         &lt;/xs:complexType>
             *       &lt;/xs:element>
             *     &lt;/xs:choice>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * </pre>
             */
            public static class FlightQuery implements Serializable
            {
                private int flightQuerySelect = -1;
                private static final int AFFINITY_CHOICE = 0;
                private static final int ORIGIN_DESTINATIONS_CHOICE = 1;
                private static final int FLIGHT_SPECIFIC_CHOICE = 2;
                private List<OriginDestination> originDestinationList = new ArrayList<OriginDestination>();
                private StayPeriodDateRangeType affinityStayPeriodDateRange;
                private StayPeriodSeasonType affinityStayPeriodSeason;
                private TravelDistancePreferencesType affinityTripDistance;
                private TravelTimePreferencesType affinityTripTime;
                private Budget budget;
                private KeywordPreferences keywordPreferences;
                private ShoppingResponseIDType affinityShoppingResponseID;
                private List<OriginDestination1> originDestinationList1s = new ArrayList<OriginDestination1>();
                private int flightSpecificSelect = -1;
                private static final int FLIGHT_SEGMENT_LIST_CHOICE = 0;
                private static final int FLIGHT_LIST_CHOICE = 1;
                private static final int FLIGHT_SPECIFIC_ORIGIN_DESTINATION_LIST_CHOICE = 2;
                private List<FlightSegment> flightSegmentList = new ArrayList<FlightSegment>();
                private List<Flight> flightList = new ArrayList<Flight>();
                private List<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination> flightSpecificOriginDestinationList = new ArrayList<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination>();

                private void setFlightQuerySelect(int choice) {
                    if (flightQuerySelect == -1) {
                        flightQuerySelect = choice;
                    } else if (flightQuerySelect != choice) {
                        throw new IllegalStateException(
                                "Need to call clearFlightQuerySelect() before changing existing choice");
                    }
                }

                /** 
                 * Clear the choice selection.
                 */
                public void clearFlightQuerySelect() {
                    flightQuerySelect = -1;
                }

                /** 
                 * Check if Affinity is current selection for choice.
                 * 
                 * @return <code>true</code> if selection, <code>false</code> if not
                 */
                public boolean ifAffinity() {
                    return flightQuerySelect == AFFINITY_CHOICE;
                }

                /** 
                 * Get the list of 'OriginDestination' element items.
                 * 
                 * @return list
                 */
                public List<OriginDestination> getOriginDestinationList() {
                    return originDestinationList;
                }

                /** 
                 * Set the list of 'OriginDestination' element items.
                 * 
                 * @param list
                 */
                public void setOriginDestinationList(
                        List<OriginDestination> list) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    originDestinationList = list;
                }

                /** 
                 * Get the 'StayPeriodDateRange' element value.
                 * 
                 * @return value
                 */
                public StayPeriodDateRangeType getAffinityStayPeriodDateRange() {
                    return affinityStayPeriodDateRange;
                }

                /** 
                 * Set the 'StayPeriodDateRange' element value.
                 * 
                 * @param affinityStayPeriodDateRange
                 */
                public void setAffinityStayPeriodDateRange(
                        StayPeriodDateRangeType affinityStayPeriodDateRange) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.affinityStayPeriodDateRange = affinityStayPeriodDateRange;
                }

                /** 
                 * Get the 'StayPeriodSeason' element value.
                 * 
                 * @return value
                 */
                public StayPeriodSeasonType getAffinityStayPeriodSeason() {
                    return affinityStayPeriodSeason;
                }

                /** 
                 * Set the 'StayPeriodSeason' element value.
                 * 
                 * @param affinityStayPeriodSeason
                 */
                public void setAffinityStayPeriodSeason(
                        StayPeriodSeasonType affinityStayPeriodSeason) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.affinityStayPeriodSeason = affinityStayPeriodSeason;
                }

                /** 
                 * Get the 'TripDistance' element value.
                 * 
                 * @return value
                 */
                public TravelDistancePreferencesType getAffinityTripDistance() {
                    return affinityTripDistance;
                }

                /** 
                 * Set the 'TripDistance' element value.
                 * 
                 * @param affinityTripDistance
                 */
                public void setAffinityTripDistance(
                        TravelDistancePreferencesType affinityTripDistance) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.affinityTripDistance = affinityTripDistance;
                }

                /** 
                 * Get the 'TripTime' element value.
                 * 
                 * @return value
                 */
                public TravelTimePreferencesType getAffinityTripTime() {
                    return affinityTripTime;
                }

                /** 
                 * Set the 'TripTime' element value.
                 * 
                 * @param affinityTripTime
                 */
                public void setAffinityTripTime(
                        TravelTimePreferencesType affinityTripTime) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.affinityTripTime = affinityTripTime;
                }

                /** 
                 * Get the 'Budget' element value.
                 * 
                 * @return value
                 */
                public Budget getBudget() {
                    return budget;
                }

                /** 
                 * Set the 'Budget' element value.
                 * 
                 * @param budget
                 */
                public void setBudget(Budget budget) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.budget = budget;
                }

                /** 
                 * Get the 'KeywordPreferences' element value.
                 * 
                 * @return value
                 */
                public KeywordPreferences getKeywordPreferences() {
                    return keywordPreferences;
                }

                /** 
                 * Set the 'KeywordPreferences' element value.
                 * 
                 * @param keywordPreferences
                 */
                public void setKeywordPreferences(
                        KeywordPreferences keywordPreferences) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.keywordPreferences = keywordPreferences;
                }

                /** 
                 * Get the 'ShoppingResponseID' element value.
                 * 
                 * @return value
                 */
                public ShoppingResponseIDType getAffinityShoppingResponseID() {
                    return affinityShoppingResponseID;
                }

                /** 
                 * Set the 'ShoppingResponseID' element value.
                 * 
                 * @param affinityShoppingResponseID
                 */
                public void setAffinityShoppingResponseID(
                        ShoppingResponseIDType affinityShoppingResponseID) {
                    setFlightQuerySelect(AFFINITY_CHOICE);
                    this.affinityShoppingResponseID = affinityShoppingResponseID;
                }

                /** 
                 * Check if OriginDestinationList1 is current selection for choice.
                 * 
                 * @return <code>true</code> if selection, <code>false</code> if not
                 */
                public boolean ifOriginDestinations() {
                    return flightQuerySelect == ORIGIN_DESTINATIONS_CHOICE;
                }

                /** 
                 * Get the list of 'OriginDestination' element items. ATTRIBUTE Query ORIGIN/ DESTINATION qualifiers and preferences.

                Note:
                1. An Attribute Query is a search specifying one or more attributes to get more focused results   Note that this node only contains Origin/ Destination Airport/ City code information which are mandatory.

                2. Other Attribute Query parameters that may be specified (in other document nodes) include Aircraft Equipment preferences, Seat types and characteristics, Meal preferences and Special Need qualifiers.
                 * 
                 * @return list
                 */
                public List<OriginDestination1> getOriginDestinationList1s() {
                    return originDestinationList1s;
                }

                /** 
                 * Set the list of 'OriginDestination' element items. ATTRIBUTE Query ORIGIN/ DESTINATION qualifiers and preferences.

                Note:
                1. An Attribute Query is a search specifying one or more attributes to get more focused results   Note that this node only contains Origin/ Destination Airport/ City code information which are mandatory.

                2. Other Attribute Query parameters that may be specified (in other document nodes) include Aircraft Equipment preferences, Seat types and characteristics, Meal preferences and Special Need qualifiers.
                 * 
                 * @param list
                 */
                public void setOriginDestinationList1s(
                        List<OriginDestination1> list) {
                    setFlightQuerySelect(ORIGIN_DESTINATIONS_CHOICE);
                    originDestinationList1s = list;
                }

                /** 
                 * Check if FlightSpecific is current selection for choice.
                 * 
                 * @return <code>true</code> if selection, <code>false</code> if not
                 */
                public boolean ifFlightSpecific() {
                    return flightQuerySelect == FLIGHT_SPECIFIC_CHOICE;
                }

                private void setFlightSpecificSelect(int choice) {
                    setFlightQuerySelect(FLIGHT_SPECIFIC_CHOICE);
                    if (flightSpecificSelect == -1) {
                        flightSpecificSelect = choice;
                    } else if (flightSpecificSelect != choice) {
                        throw new IllegalStateException(
                                "Need to call clearFlightSpecificSelect() before changing existing choice");
                    }
                }

                /** 
                 * Clear the choice selection.
                 */
                public void clearFlightSpecificSelect() {
                    flightSpecificSelect = -1;
                }

                /** 
                 * Check if FlightSegmentList is current selection for choice.
                 * 
                 * @return <code>true</code> if selection, <code>false</code> if not
                 */
                public boolean ifFlightSegmentList() {
                    return flightSpecificSelect == FLIGHT_SEGMENT_LIST_CHOICE;
                }

                /** 
                 * Get the list of 'FlightSegment' element items.
                 * 
                 * @return list
                 */
                public List<FlightSegment> getFlightSegmentList() {
                    return flightSegmentList;
                }

                /** 
                 * Set the list of 'FlightSegment' element items.
                 * 
                 * @param list
                 */
                public void setFlightSegmentList(List<FlightSegment> list) {
                    setFlightSpecificSelect(FLIGHT_SEGMENT_LIST_CHOICE);
                    flightSegmentList = list;
                }

                /** 
                 * Check if FlightList is current selection for choice.
                 * 
                 * @return <code>true</code> if selection, <code>false</code> if not
                 */
                public boolean ifFlightList() {
                    return flightSpecificSelect == FLIGHT_LIST_CHOICE;
                }

                /** 
                 * Get the list of 'Flight' element items.
                 * 
                 * @return list
                 */
                public List<Flight> getFlightList() {
                    return flightList;
                }

                /** 
                 * Set the list of 'Flight' element items.
                 * 
                 * @param list
                 */
                public void setFlightList(List<Flight> list) {
                    setFlightSpecificSelect(FLIGHT_LIST_CHOICE);
                    flightList = list;
                }

                /** 
                 * Check if FlightSpecificOriginDestinationList is current selection for choice.
                 * 
                 * @return <code>true</code> if selection, <code>false</code> if not
                 */
                public boolean ifFlightSpecificOriginDestinationList() {
                    return flightSpecificSelect == FLIGHT_SPECIFIC_ORIGIN_DESTINATION_LIST_CHOICE;
                }

                /** 
                 * Get the list of 'OriginDestination' element items.
                 * 
                 * @return list
                 */
                public List<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination> getFlightSpecificOriginDestinationList() {
                    return flightSpecificOriginDestinationList;
                }

                /** 
                 * Set the list of 'OriginDestination' element items.
                 * 
                 * @param list
                 */
                public void setFlightSpecificOriginDestinationList(
                        List<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination> list) {
                    setFlightSpecificSelect(FLIGHT_SPECIFIC_ORIGIN_DESTINATION_LIST_CHOICE);
                    flightSpecificOriginDestinationList = list;
                }
                /** 
                 * Flight origin/ destination information.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="OriginDestination" maxOccurs="unbounded">
                 *   &lt;xs:complexType>
                 *     &lt;xs:complexContent>
                 *       &lt;xs:extension base="ns:AssociatedObjectBaseType">
                 *         &lt;xs:sequence>
                 *           &lt;xs:element ref="ns:AirportCityDeparture" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:CountryDeparture" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:ReferencePointDeparture" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:StateProvDeparture" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:AirportCityArrival" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:CountryArrival" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:ReferencePointArrival" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:StateProvArrival" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:SegMaxTimePreferences" minOccurs="0"/>
                 *         &lt;/xs:sequence>
                 *       &lt;/xs:extension>
                 *     &lt;/xs:complexContent>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class OriginDestination
                        extends
                            AssociatedObjectBaseType implements Serializable
                {
                    private FltDepartQualifiedQueryType airportCityDeparture;
                    private AffinityCountryDepartType countryDeparture;
                    private AffinityReferencePointDepartType referencePointDeparture;
                    private AffinityStateProvDepartType stateProvDeparture;
                    private FltArriveQualifiedQueryType airportCityArrival;
                    private AffinityCountryArriveType countryArrival;
                    private AffinityReferencePointArriveType referencePointArrival;
                    private AffinityStateProvArriveType stateProvArrival;
                    private FltSegmentMaxTimePreferencesType segMaxTimePreferences;

                    /** 
                     * Get the 'AirportCityDeparture' element value.
                     * 
                     * @return value
                     */
                    public FltDepartQualifiedQueryType getAirportCityDeparture() {
                        return airportCityDeparture;
                    }

                    /** 
                     * Set the 'AirportCityDeparture' element value.
                     * 
                     * @param airportCityDeparture
                     */
                    public void setAirportCityDeparture(
                            FltDepartQualifiedQueryType airportCityDeparture) {
                        this.airportCityDeparture = airportCityDeparture;
                    }

                    /** 
                     * Get the 'CountryDeparture' element value.
                     * 
                     * @return value
                     */
                    public AffinityCountryDepartType getCountryDeparture() {
                        return countryDeparture;
                    }

                    /** 
                     * Set the 'CountryDeparture' element value.
                     * 
                     * @param countryDeparture
                     */
                    public void setCountryDeparture(
                            AffinityCountryDepartType countryDeparture) {
                        this.countryDeparture = countryDeparture;
                    }

                    /** 
                     * Get the 'ReferencePointDeparture' element value.
                     * 
                     * @return value
                     */
                    public AffinityReferencePointDepartType getReferencePointDeparture() {
                        return referencePointDeparture;
                    }

                    /** 
                     * Set the 'ReferencePointDeparture' element value.
                     * 
                     * @param referencePointDeparture
                     */
                    public void setReferencePointDeparture(
                            AffinityReferencePointDepartType referencePointDeparture) {
                        this.referencePointDeparture = referencePointDeparture;
                    }

                    /** 
                     * Get the 'StateProvDeparture' element value.
                     * 
                     * @return value
                     */
                    public AffinityStateProvDepartType getStateProvDeparture() {
                        return stateProvDeparture;
                    }

                    /** 
                     * Set the 'StateProvDeparture' element value.
                     * 
                     * @param stateProvDeparture
                     */
                    public void setStateProvDeparture(
                            AffinityStateProvDepartType stateProvDeparture) {
                        this.stateProvDeparture = stateProvDeparture;
                    }

                    /** 
                     * Get the 'AirportCityArrival' element value.
                     * 
                     * @return value
                     */
                    public FltArriveQualifiedQueryType getAirportCityArrival() {
                        return airportCityArrival;
                    }

                    /** 
                     * Set the 'AirportCityArrival' element value.
                     * 
                     * @param airportCityArrival
                     */
                    public void setAirportCityArrival(
                            FltArriveQualifiedQueryType airportCityArrival) {
                        this.airportCityArrival = airportCityArrival;
                    }

                    /** 
                     * Get the 'CountryArrival' element value.
                     * 
                     * @return value
                     */
                    public AffinityCountryArriveType getCountryArrival() {
                        return countryArrival;
                    }

                    /** 
                     * Set the 'CountryArrival' element value.
                     * 
                     * @param countryArrival
                     */
                    public void setCountryArrival(
                            AffinityCountryArriveType countryArrival) {
                        this.countryArrival = countryArrival;
                    }

                    /** 
                     * Get the 'ReferencePointArrival' element value.
                     * 
                     * @return value
                     */
                    public AffinityReferencePointArriveType getReferencePointArrival() {
                        return referencePointArrival;
                    }

                    /** 
                     * Set the 'ReferencePointArrival' element value.
                     * 
                     * @param referencePointArrival
                     */
                    public void setReferencePointArrival(
                            AffinityReferencePointArriveType referencePointArrival) {
                        this.referencePointArrival = referencePointArrival;
                    }

                    /** 
                     * Get the 'StateProvArrival' element value.
                     * 
                     * @return value
                     */
                    public AffinityStateProvArriveType getStateProvArrival() {
                        return stateProvArrival;
                    }

                    /** 
                     * Set the 'StateProvArrival' element value.
                     * 
                     * @param stateProvArrival
                     */
                    public void setStateProvArrival(
                            AffinityStateProvArriveType stateProvArrival) {
                        this.stateProvArrival = stateProvArrival;
                    }

                    /** 
                     * Get the 'SegMaxTimePreferences' element value.
                     * 
                     * @return value
                     */
                    public FltSegmentMaxTimePreferencesType getSegMaxTimePreferences() {
                        return segMaxTimePreferences;
                    }

                    /** 
                     * Set the 'SegMaxTimePreferences' element value.
                     * 
                     * @param segMaxTimePreferences
                     */
                    public void setSegMaxTimePreferences(
                            FltSegmentMaxTimePreferencesType segMaxTimePreferences) {
                        this.segMaxTimePreferences = segMaxTimePreferences;
                    }
                }
                /** 
                 * Trip Budget.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Budget" minOccurs="0">
                 *   &lt;xs:complexType>
                 *     &lt;xs:choice>
                 *       &lt;xs:element ref="ns:GroupBudget"/>
                 *       &lt;xs:element ref="ns:PerPersonBudget"/>
                 *     &lt;/xs:choice>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class Budget implements Serializable
                {
                    private int budgetSelect = -1;
                    private static final int GROUP_BUDGET_CHOICE = 0;
                    private static final int PER_PERSON_BUDGET_CHOICE = 1;
                    private GroupBudgetType groupBudget;
                    private PersonBudgetType perPersonBudget;

                    private void setBudgetSelect(int choice) {
                        if (budgetSelect == -1) {
                            budgetSelect = choice;
                        } else if (budgetSelect != choice) {
                            throw new IllegalStateException(
                                    "Need to call clearBudgetSelect() before changing existing choice");
                        }
                    }

                    /** 
                     * Clear the choice selection.
                     */
                    public void clearBudgetSelect() {
                        budgetSelect = -1;
                    }

                    /** 
                     * Check if GroupBudget is current selection for choice.
                     * 
                     * @return <code>true</code> if selection, <code>false</code> if not
                     */
                    public boolean ifGroupBudget() {
                        return budgetSelect == GROUP_BUDGET_CHOICE;
                    }

                    /** 
                     * Get the 'GroupBudget' element value.
                     * 
                     * @return value
                     */
                    public GroupBudgetType getGroupBudget() {
                        return groupBudget;
                    }

                    /** 
                     * Set the 'GroupBudget' element value.
                     * 
                     * @param groupBudget
                     */
                    public void setGroupBudget(GroupBudgetType groupBudget) {
                        setBudgetSelect(GROUP_BUDGET_CHOICE);
                        this.groupBudget = groupBudget;
                    }

                    /** 
                     * Check if PerPersonBudget is current selection for choice.
                     * 
                     * @return <code>true</code> if selection, <code>false</code> if not
                     */
                    public boolean ifPerPersonBudget() {
                        return budgetSelect == PER_PERSON_BUDGET_CHOICE;
                    }

                    /** 
                     * Get the 'PerPersonBudget' element value.
                     * 
                     * @return value
                     */
                    public PersonBudgetType getPerPersonBudget() {
                        return perPersonBudget;
                    }

                    /** 
                     * Set the 'PerPersonBudget' element value.
                     * 
                     * @param perPersonBudget
                     */
                    public void setPerPersonBudget(
                            PersonBudgetType perPersonBudget) {
                        setBudgetSelect(PER_PERSON_BUDGET_CHOICE);
                        this.perPersonBudget = perPersonBudget;
                    }
                }
                /** 
                 * Affinity Query Keyword(s) with optional preference level.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="KeywordPreferences" minOccurs="0">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:element name="KeywordPreference" maxOccurs="unbounded">
                 *         &lt;!-- Reference to inner class KeywordPreference -->
                 *       &lt;/xs:element>
                 *     &lt;/xs:sequence>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class KeywordPreferences implements Serializable
                {
                    private List<KeywordPreference> keywordPreferenceList = new ArrayList<KeywordPreference>();

                    /** 
                     * Get the list of 'KeywordPreference' element items.
                     * 
                     * @return list
                     */
                    public List<KeywordPreference> getKeywordPreferenceList() {
                        return keywordPreferenceList;
                    }

                    /** 
                     * Set the list of 'KeywordPreference' element items.
                     * 
                     * @param list
                     */
                    public void setKeywordPreferenceList(
                            List<KeywordPreference> list) {
                        keywordPreferenceList = list;
                    }
                    /** 
                     * Keyword/ Value Search Criteria.

                    Note: Keyword-Value pairs may be used to specify a variety of affinity qualifiers, such as Climate, Activity, and Point of Interest.
                     * 
                     * Schema fragment(s) for this class:
                     * <pre>
                     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="KeywordPreference" maxOccurs="unbounded">
                     *   &lt;xs:complexType>
                     *     &lt;xs:complexContent>
                     *       &lt;xs:extension base="ns:KeyWordType">
                     *         &lt;xs:attributeGroup ref="ns:PreferenceAttrGroup"/>
                     *       &lt;/xs:extension>
                     *     &lt;/xs:complexContent>
                     *   &lt;/xs:complexType>
                     * &lt;/xs:element>
                     * </pre>
                     */
                    public static class KeywordPreference extends KeyWordType
                        implements Serializable
                    {
                        private PreferenceAttrGroup preferenceAttrGroup;

                        /** 
                         * Get the 'PreferenceAttrGroup' attributeGroup value.
                         * 
                         * @return value
                         */
                        public PreferenceAttrGroup getPreferenceAttrGroup() {
                            return preferenceAttrGroup;
                        }

                        /** 
                         * Set the 'PreferenceAttrGroup' attributeGroup value.
                         * 
                         * @param preferenceAttrGroup
                         */
                        public void setPreferenceAttrGroup(
                                PreferenceAttrGroup preferenceAttrGroup) {
                            this.preferenceAttrGroup = preferenceAttrGroup;
                        }
                    }
                }
                /** 
                 * Flight origin/ destination information.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="OriginDestination" maxOccurs="unbounded">
                 *   &lt;xs:complexType>
                 *     &lt;xs:complexContent>
                 *       &lt;xs:extension base="ns:AssociatedObjectBaseType">
                 *         &lt;xs:sequence>
                 *           &lt;xs:element ref="ns:Departure"/>
                 *           &lt;xs:element ref="ns:Arrival"/>
                 *           &lt;xs:element ref="ns:SegMaxTimePreferences" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:ShoppingResponseID" minOccurs="0"/>
                 *           &lt;xs:element ref="ns:MarketingCarrierAirline" minOccurs="0"/>
                 *           &lt;xs:element name="CalendarDates" minOccurs="0">
                 *             &lt;!-- Reference to inner class CalendarDates -->
                 *           &lt;/xs:element>
                 *         &lt;/xs:sequence>
                 *         &lt;xs:attribute type="xs:anySimpleType" name="OriginDestinationKey"/>
                 *       &lt;/xs:extension>
                 *     &lt;/xs:complexContent>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class OriginDestination1
                        extends
                            AssociatedObjectBaseType implements Serializable
                {
                    private FlightDepartureType departure;
                    private FlightArrivalType arrival;
                    private FltSegmentMaxTimePreferencesType segMaxTimePreferences;
                    private ShoppingResponseIDType shoppingResponseID;
                    private MarketingCarrierType marketingCarrierAirline;
                    private CalendarDates calendarDates;
                    private String originDestinationKey;

                    /** 
                     * Get the 'Departure' element value. FLIGHT DEPARTURE (ORIGIN).
                     * 
                     * @return value
                     */
                    public FlightDepartureType getDeparture() {
                        return departure;
                    }

                    /** 
                     * Set the 'Departure' element value. FLIGHT DEPARTURE (ORIGIN).
                     * 
                     * @param departure
                     */
                    public void setDeparture(FlightDepartureType departure) {
                        this.departure = departure;
                    }

                    /** 
                     * Get the 'Arrival' element value. FLIGHT ARRIVAL (DESTINATION).
                     * 
                     * @return value
                     */
                    public FlightArrivalType getArrival() {
                        return arrival;
                    }

                    /** 
                     * Set the 'Arrival' element value. FLIGHT ARRIVAL (DESTINATION).
                     * 
                     * @param arrival
                     */
                    public void setArrival(FlightArrivalType arrival) {
                        this.arrival = arrival;
                    }

                    /** 
                     * Get the 'SegMaxTimePreferences' element value.
                     * 
                     * @return value
                     */
                    public FltSegmentMaxTimePreferencesType getSegMaxTimePreferences() {
                        return segMaxTimePreferences;
                    }

                    /** 
                     * Set the 'SegMaxTimePreferences' element value.
                     * 
                     * @param segMaxTimePreferences
                     */
                    public void setSegMaxTimePreferences(
                            FltSegmentMaxTimePreferencesType segMaxTimePreferences) {
                        this.segMaxTimePreferences = segMaxTimePreferences;
                    }

                    /** 
                     * Get the 'ShoppingResponseID' element value. Flight-related shopping response IDs, including the unique (master) Shopping Session ID, Offer IDs, Offer Item IDs and Service IDs.
                     * 
                     * @return value
                     */
                    public ShoppingResponseIDType getShoppingResponseID() {
                        return shoppingResponseID;
                    }

                    /** 
                     * Set the 'ShoppingResponseID' element value. Flight-related shopping response IDs, including the unique (master) Shopping Session ID, Offer IDs, Offer Item IDs and Service IDs.
                     * 
                     * @param shoppingResponseID
                     */
                    public void setShoppingResponseID(
                            ShoppingResponseIDType shoppingResponseID) {
                        this.shoppingResponseID = shoppingResponseID;
                    }

                    /** 
                     * Get the 'MarketingCarrierAirline' element value.
                     * 
                     * @return value
                     */
                    public MarketingCarrierType getMarketingCarrierAirline() {
                        return marketingCarrierAirline;
                    }

                    /** 
                     * Set the 'MarketingCarrierAirline' element value.
                     * 
                     * @param marketingCarrierAirline
                     */
                    public void setMarketingCarrierAirline(
                            MarketingCarrierType marketingCarrierAirline) {
                        this.marketingCarrierAirline = marketingCarrierAirline;
                    }

                    /** 
                     * Get the 'CalendarDates' element value.
                     * 
                     * @return value
                     */
                    public CalendarDates getCalendarDates() {
                        return calendarDates;
                    }

                    /** 
                     * Set the 'CalendarDates' element value.
                     * 
                     * @param calendarDates
                     */
                    public void setCalendarDates(CalendarDates calendarDates) {
                        this.calendarDates = calendarDates;
                    }

                    /** 
                     * Get the 'OriginDestinationKey' attribute value. A globally unique identifier for this OriginDestination instance. Example: origDest1
                     * 
                     * @return value
                     */
                    public String getOriginDestinationKey() {
                        return originDestinationKey;
                    }

                    /** 
                     * Set the 'OriginDestinationKey' attribute value. A globally unique identifier for this OriginDestination instance. Example: origDest1
                     * 
                     * @param originDestinationKey
                     */
                    public void setOriginDestinationKey(
                            String originDestinationKey) {
                        this.originDestinationKey = originDestinationKey;
                    }
                    /** 
                     * Indicates a calendar search request with days before and/or days after the specified date.
                     * 
                     * Schema fragment(s) for this class:
                     * <pre>
                     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="CalendarDates" minOccurs="0">
                     *   &lt;xs:complexType>
                     *     &lt;xs:attribute type="xs:int" name="DaysBefore"/>
                     *     &lt;xs:attribute type="xs:int" name="DaysAfter"/>
                     *   &lt;/xs:complexType>
                     * &lt;/xs:element>
                     * </pre>
                     */
                    public static class CalendarDates implements Serializable
                    {
                        private Integer daysBefore;
                        private Integer daysAfter;

                        /** 
                         * Get the 'DaysBefore' attribute value. Number of days  befiore the specified date.
                         * 
                         * @return value
                         */
                        public Integer getDaysBefore() {
                            return daysBefore;
                        }

                        /** 
                         * Set the 'DaysBefore' attribute value. Number of days  befiore the specified date.
                         * 
                         * @param daysBefore
                         */
                        public void setDaysBefore(Integer daysBefore) {
                            this.daysBefore = daysBefore;
                        }

                        /** 
                         * Get the 'DaysAfter' attribute value. Number  of days after the specified date.
                         * 
                         * @return value
                         */
                        public Integer getDaysAfter() {
                            return daysAfter;
                        }

                        /** 
                         * Set the 'DaysAfter' attribute value. Number  of days after the specified date.
                         * 
                         * @param daysAfter
                         */
                        public void setDaysAfter(Integer daysAfter) {
                            this.daysAfter = daysAfter;
                        }
                    }
                }
                /** 
                 * FlighSegment information. Contains details flight segment information.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="FlightSegment" maxOccurs="unbounded">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:element ref="ns:Departure"/>
                 *       &lt;xs:element ref="ns:Arrival"/>
                 *       &lt;xs:element type="ns:MarketingCarrierFlightType" name="MarketingAirline"/>
                 *       &lt;xs:element type="ns:OperatingCarrierFlightType" name="OperatingAirline" minOccurs="0"/>
                 *     &lt;/xs:sequence>
                 *     &lt;xs:attribute type="xs:int" use="required" name="SegmentKey"/>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class FlightSegment implements Serializable
                {
                    private FlightDepartureType departure;
                    private FlightArrivalType arrival;
                    private MarketingCarrierFlightType marketingAirline;
                    private OperatingCarrierFlightType operatingAirline;
                    private Integer segmentKey;

                    /** 
                     * Get the 'Departure' element value.
                     * 
                     * @return value
                     */
                    public FlightDepartureType getDeparture() {
                        return departure;
                    }

                    /** 
                     * Set the 'Departure' element value.
                     * 
                     * @param departure
                     */
                    public void setDeparture(FlightDepartureType departure) {
                        this.departure = departure;
                    }

                    /** 
                     * Get the 'Arrival' element value.
                     * 
                     * @return value
                     */
                    public FlightArrivalType getArrival() {
                        return arrival;
                    }

                    /** 
                     * Set the 'Arrival' element value.
                     * 
                     * @param arrival
                     */
                    public void setArrival(FlightArrivalType arrival) {
                        this.arrival = arrival;
                    }

                    /** 
                     * Get the 'MarketingAirline' element value. Marketing Carrier information.
                     * 
                     * @return value
                     */
                    public MarketingCarrierFlightType getMarketingAirline() {
                        return marketingAirline;
                    }

                    /** 
                     * Set the 'MarketingAirline' element value. Marketing Carrier information.
                     * 
                     * @param marketingAirline
                     */
                    public void setMarketingAirline(
                            MarketingCarrierFlightType marketingAirline) {
                        this.marketingAirline = marketingAirline;
                    }

                    /** 
                     * Get the 'OperatingAirline' element value. Operating Carrier information.

                    Note: This is the carrier that holds the Air Operator’s Certificate for the aircraft used for that flight.
                     * 
                     * @return value
                     */
                    public OperatingCarrierFlightType getOperatingAirline() {
                        return operatingAirline;
                    }

                    /** 
                     * Set the 'OperatingAirline' element value. Operating Carrier information.

                    Note: This is the carrier that holds the Air Operator’s Certificate for the aircraft used for that flight.
                     * 
                     * @param operatingAirline
                     */
                    public void setOperatingAirline(
                            OperatingCarrierFlightType operatingAirline) {
                        this.operatingAirline = operatingAirline;
                    }

                    /** 
                     * Get the 'SegmentKey' attribute value. A globally unique identifier for this Flight Segment instance. Example: segment1
                     * 
                     * @return value
                     */
                    public Integer getSegmentKey() {
                        return segmentKey;
                    }

                    /** 
                     * Set the 'SegmentKey' attribute value. A globally unique identifier for this Flight Segment instance. Example: segment1
                     * 
                     * @param segmentKey
                     */
                    public void setSegmentKey(Integer segmentKey) {
                        this.segmentKey = segmentKey;
                    }
                }
                /** 
                 * Flight information. Flight can be made of one to multiple FlightSegment elements. For example Flight can be made of flight segment BA123 and BA34.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Flight" maxOccurs="unbounded">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:element type="ns:TotalJourneyType" name="Journey" minOccurs="0"/>
                 *       &lt;xs:element ref="ns:SegmentReferences"/>
                 *     &lt;/xs:sequence>
                 *     &lt;xs:attributeGroup ref="ns:ObjAssociationAttrGroup"/>
                 *     &lt;xs:attribute type="xs:string" name="FlightKey"/>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * 
                 * &lt;xs:attributeGroup xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="ObjAssociationAttrGroup">
                 *   &lt;xs:attribute type="xs:string" use="optional" name="refs"/>
                 * &lt;/xs:attributeGroup>
                 * </pre>
                 */
                public static class Flight implements Serializable
                {
                    private TotalJourneyType journey;
                    private SegmentReferences segmentReferences;
                    private String refs;
                    private String flightKey;

                    /** 
                     * Get the 'Journey' element value. Flight journey information.
                     * 
                     * @return value
                     */
                    public TotalJourneyType getJourney() {
                        return journey;
                    }

                    /** 
                     * Set the 'Journey' element value. Flight journey information.
                     * 
                     * @param journey
                     */
                    public void setJourney(TotalJourneyType journey) {
                        this.journey = journey;
                    }

                    /** 
                     * Get the 'SegmentReferences' element value.
                     * 
                     * @return value
                     */
                    public SegmentReferences getSegmentReferences() {
                        return segmentReferences;
                    }

                    /** 
                     * Set the 'SegmentReferences' element value.
                     * 
                     * @param segmentReferences
                     */
                    public void setSegmentReferences(
                            SegmentReferences segmentReferences) {
                        this.segmentReferences = segmentReferences;
                    }

                    /** 
                     * Get the 'refs' attribute value. Object association(s) between this object and any other object or metadata instance(s).
                     * 
                     * @return value
                     */
                    public String getRefs() {
                        return refs;
                    }

                    /** 
                     * Set the 'refs' attribute value. Object association(s) between this object and any other object or metadata instance(s).
                     * 
                     * @param refs
                     */
                    public void setRefs(String refs) {
                        this.refs = refs;
                    }

                    /** 
                     * Get the 'FlightKey' attribute value. A globally unique identifier for this Flight instance. Example: flight1
                     * 
                     * @return value
                     */
                    public String getFlightKey() {
                        return flightKey;
                    }

                    /** 
                     * Set the 'FlightKey' attribute value. A globally unique identifier for this Flight instance. Example: flight1
                     * 
                     * @param flightKey
                     */
                    public void setFlightKey(String flightKey) {
                        this.flightKey = flightKey;
                    }
                }
            }
            /** 
             * Qualifier choices.

            Notes: 
            1. These qualifiers may be specified for (e.g. apply to) both Attribute and Affinity queries.
            2. These capabilities may not be supported by all NDC-enabled systems or may be supported by bilateral agreement only. If supported, the technical implementation of the capability is not dictated by this specification. 
             * 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Qualifier" minOccurs="0" maxOccurs="unbounded">
             *   &lt;xs:complexType>
             *     &lt;xs:sequence>
             *       &lt;xs:element ref="ns:BaggagePricingQualifier" minOccurs="0"/>
             *       &lt;xs:element ref="ns:ExistingOrderQualifier" minOccurs="0"/>
             *       &lt;xs:element ref="ns:PaymentCardQualifiers" minOccurs="0" maxOccurs="unbounded"/>
             *       &lt;xs:element ref="ns:ProgramQualifiers" minOccurs="0"/>
             *       &lt;xs:element ref="ns:PromotionQualifiers" minOccurs="0"/>
             *       &lt;xs:element ref="ns:SeatQualifier" minOccurs="0"/>
             *       &lt;xs:element ref="ns:ServiceQualifier" minOccurs="0"/>
             *       &lt;xs:element ref="ns:SocialMediaQualifiers" minOccurs="0"/>
             *       &lt;xs:element ref="ns:SpecialFareQualifiers" minOccurs="0"/>
             *       &lt;xs:element ref="ns:SpecialNeedQualifiers" minOccurs="0"/>
             *       &lt;xs:element ref="ns:TripPurposeQualifier" minOccurs="0"/>
             *       &lt;xs:element name="FOPQualifier" minOccurs="0">
             *         &lt;!-- Reference to inner class FOPQualifier -->
             *       &lt;/xs:element>
             *     &lt;/xs:sequence>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * </pre>
             */
            public static class Qualifier implements Serializable
            {
                private BaggagePricingQualifierType baggagePricingQualifier;
                private ExistingOrderQualifier existingOrderQualifier;
                private List<CardQualifierType> paymentCardQualifierList = new ArrayList<CardQualifierType>();
                private ProgramQualifiers programQualifiers;
                private PromotionQualifiers promotionQualifiers;
                private SeatQualifier seatQualifier;
                private ServiceQualifierPriceType serviceQualifier;
                private SocialMediaQualifierType socialMediaQualifiers;
                private FareQualifierType specialFareQualifiers;
                private SpecialType specialNeedQualifiers;
                private TripPurposeQualifier tripPurposeQualifier;
                private FOPQualifier FOPQualifier1;

                /** 
                 * Get the 'BaggagePricingQualifier' element value.
                 * 
                 * @return value
                 */
                public BaggagePricingQualifierType getBaggagePricingQualifier() {
                    return baggagePricingQualifier;
                }

                /** 
                 * Set the 'BaggagePricingQualifier' element value.
                 * 
                 * @param baggagePricingQualifier
                 */
                public void setBaggagePricingQualifier(
                        BaggagePricingQualifierType baggagePricingQualifier) {
                    this.baggagePricingQualifier = baggagePricingQualifier;
                }

                /** 
                 * Get the 'ExistingOrderQualifier' element value.
                 * 
                 * @return value
                 */
                public ExistingOrderQualifier getExistingOrderQualifier() {
                    return existingOrderQualifier;
                }

                /** 
                 * Set the 'ExistingOrderQualifier' element value.
                 * 
                 * @param existingOrderQualifier
                 */
                public void setExistingOrderQualifier(
                        ExistingOrderQualifier existingOrderQualifier) {
                    this.existingOrderQualifier = existingOrderQualifier;
                }

                /** 
                 * Get the list of 'PaymentCardQualifiers' element items.
                 * 
                 * @return list
                 */
                public List<CardQualifierType> getPaymentCardQualifierList() {
                    return paymentCardQualifierList;
                }

                /** 
                 * Set the list of 'PaymentCardQualifiers' element items.
                 * 
                 * @param list
                 */
                public void setPaymentCardQualifierList(
                        List<CardQualifierType> list) {
                    paymentCardQualifierList = list;
                }

                /** 
                 * Get the 'ProgramQualifiers' element value.
                 * 
                 * @return value
                 */
                public ProgramQualifiers getProgramQualifiers() {
                    return programQualifiers;
                }

                /** 
                 * Set the 'ProgramQualifiers' element value.
                 * 
                 * @param programQualifiers
                 */
                public void setProgramQualifiers(
                        ProgramQualifiers programQualifiers) {
                    this.programQualifiers = programQualifiers;
                }

                /** 
                 * Get the 'PromotionQualifiers' element value.
                 * 
                 * @return value
                 */
                public PromotionQualifiers getPromotionQualifiers() {
                    return promotionQualifiers;
                }

                /** 
                 * Set the 'PromotionQualifiers' element value.
                 * 
                 * @param promotionQualifiers
                 */
                public void setPromotionQualifiers(
                        PromotionQualifiers promotionQualifiers) {
                    this.promotionQualifiers = promotionQualifiers;
                }

                /** 
                 * Get the 'SeatQualifier' element value.
                 * 
                 * @return value
                 */
                public SeatQualifier getSeatQualifier() {
                    return seatQualifier;
                }

                /** 
                 * Set the 'SeatQualifier' element value.
                 * 
                 * @param seatQualifier
                 */
                public void setSeatQualifier(SeatQualifier seatQualifier) {
                    this.seatQualifier = seatQualifier;
                }

                /** 
                 * Get the 'ServiceQualifier' element value.
                 * 
                 * @return value
                 */
                public ServiceQualifierPriceType getServiceQualifier() {
                    return serviceQualifier;
                }

                /** 
                 * Set the 'ServiceQualifier' element value.
                 * 
                 * @param serviceQualifier
                 */
                public void setServiceQualifier(
                        ServiceQualifierPriceType serviceQualifier) {
                    this.serviceQualifier = serviceQualifier;
                }

                /** 
                 * Get the 'SocialMediaQualifiers' element value.
                 * 
                 * @return value
                 */
                public SocialMediaQualifierType getSocialMediaQualifiers() {
                    return socialMediaQualifiers;
                }

                /** 
                 * Set the 'SocialMediaQualifiers' element value.
                 * 
                 * @param socialMediaQualifiers
                 */
                public void setSocialMediaQualifiers(
                        SocialMediaQualifierType socialMediaQualifiers) {
                    this.socialMediaQualifiers = socialMediaQualifiers;
                }

                /** 
                 * Get the 'SpecialFareQualifiers' element value.
                 * 
                 * @return value
                 */
                public FareQualifierType getSpecialFareQualifiers() {
                    return specialFareQualifiers;
                }

                /** 
                 * Set the 'SpecialFareQualifiers' element value.
                 * 
                 * @param specialFareQualifiers
                 */
                public void setSpecialFareQualifiers(
                        FareQualifierType specialFareQualifiers) {
                    this.specialFareQualifiers = specialFareQualifiers;
                }

                /** 
                 * Get the 'SpecialNeedQualifiers' element value.
                 * 
                 * @return value
                 */
                public SpecialType getSpecialNeedQualifiers() {
                    return specialNeedQualifiers;
                }

                /** 
                 * Set the 'SpecialNeedQualifiers' element value.
                 * 
                 * @param specialNeedQualifiers
                 */
                public void setSpecialNeedQualifiers(
                        SpecialType specialNeedQualifiers) {
                    this.specialNeedQualifiers = specialNeedQualifiers;
                }

                /** 
                 * Get the 'TripPurposeQualifier' element value.
                 * 
                 * @return value
                 */
                public TripPurposeQualifier getTripPurposeQualifier() {
                    return tripPurposeQualifier;
                }

                /** 
                 * Set the 'TripPurposeQualifier' element value.
                 * 
                 * @param tripPurposeQualifier
                 */
                public void setTripPurposeQualifier(
                        TripPurposeQualifier tripPurposeQualifier) {
                    this.tripPurposeQualifier = tripPurposeQualifier;
                }

                /** 
                 * Get the 'FOPQualifier' element value.
                 * 
                 * @return value
                 */
                public FOPQualifier getFOPQualifier1() {
                    return FOPQualifier1;
                }

                /** 
                 * Set the 'FOPQualifier' element value.
                 * 
                 * @param FOPQualifier1
                 */
                public void setFOPQualifier1(FOPQualifier FOPQualifier1) {
                    this.FOPQualifier1 = FOPQualifier1;
                }
                /** 
                 * Contains Forms of Payment type and may include a reference to a passenger.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="FOPQualifier" minOccurs="0">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:element type="xs:string" name="PassengerReferences" minOccurs="0"/>
                 *       &lt;xs:element type="xs:string" name="Type" maxOccurs="unbounded"/>
                 *     &lt;/xs:sequence>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class FOPQualifier implements Serializable
                {
                    private String passengerReferences;
                    private List<String> typeList = new ArrayList<String>();

                    /** 
                     * Get the 'PassengerReferences' element value. Association to multiple Passenger instance(s). Example: Passenger1 Passenger2

                    Note: May be Recognized or Anonymous Passenger (by PTC) instance(s).
                     * 
                     * @return value
                     */
                    public String getPassengerReferences() {
                        return passengerReferences;
                    }

                    /** 
                     * Set the 'PassengerReferences' element value. Association to multiple Passenger instance(s). Example: Passenger1 Passenger2

                    Note: May be Recognized or Anonymous Passenger (by PTC) instance(s).
                     * 
                     * @param passengerReferences
                     */
                    public void setPassengerReferences(
                            String passengerReferences) {
                        this.passengerReferences = passengerReferences;
                    }

                    /** 
                     * Get the list of 'Type' element items. The form of payment type. Examples: Credit Card, Cash.

                    Encoding Scheme: IATA PADIS code list data element [9888] Form of Payment Identification
                     * 
                     * @return list
                     */
                    public List<String> getTypeList() {
                        return typeList;
                    }

                    /** 
                     * Set the list of 'Type' element items. The form of payment type. Examples: Credit Card, Cash.

                    Encoding Scheme: IATA PADIS code list data element [9888] Form of Payment Identification
                     * 
                     * @param list
                     */
                    public void setTypeList(List<String> list) {
                        typeList = list;
                    }
                }
            }
            /** 
             * Preference choices.

            Notes: 
            1. These preferences may be specified for (e.g. apply to) both Attribute and Affinity queries.
            2. These capabilities may not be supported by all NDC-enabled systems or may be supported by bilateral agreement only. If supported, the technical implementation of the capability is not dictated by this specification. 
             * 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Preference" minOccurs="0">
             *   &lt;xs:complexType>
             *     &lt;xs:sequence>
             *       &lt;xs:element ref="ns:AirlinePreferences" minOccurs="0"/>
             *       &lt;xs:element ref="ns:AlliancePreferences" minOccurs="0"/>
             *       &lt;xs:element ref="ns:FarePreferences" minOccurs="0"/>
             *       &lt;xs:element ref="ns:FlightPreferences" minOccurs="0"/>
             *       &lt;xs:element ref="ns:PricingMethodPreference" minOccurs="0"/>
             *       &lt;xs:element ref="ns:SegMaxTimePreferences" minOccurs="0"/>
             *       &lt;xs:element ref="ns:ServicePricingOnlyPreference" minOccurs="0"/>
             *       &lt;xs:element ref="ns:TransferPreferences" minOccurs="0"/>
             *       &lt;xs:element ref="ns:CabinPreferences" minOccurs="0"/>
             *     &lt;/xs:sequence>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * </pre>
             */
            public static class Preference implements Serializable
            {
                private AirlinePreferencesType airlinePreferences;
                private AlliancePreferencesType alliancePreferences;
                private FarePreferencesType farePreferences;
                private FlightPreferencesType flightPreferences;
                private BestPricingPreferencesType pricingMethodPreference;
                private FltSegmentMaxTimePreferencesType segMaxTimePreferences;
                private ServicePricingOnlyPreference servicePricingOnlyPreference;
                private TransferPreferencesType transferPreferences;
                private CabinPreferencesType cabinPreferences;

                /** 
                 * Get the 'AirlinePreferences' element value.
                 * 
                 * @return value
                 */
                public AirlinePreferencesType getAirlinePreferences() {
                    return airlinePreferences;
                }

                /** 
                 * Set the 'AirlinePreferences' element value.
                 * 
                 * @param airlinePreferences
                 */
                public void setAirlinePreferences(
                        AirlinePreferencesType airlinePreferences) {
                    this.airlinePreferences = airlinePreferences;
                }

                /** 
                 * Get the 'AlliancePreferences' element value.
                 * 
                 * @return value
                 */
                public AlliancePreferencesType getAlliancePreferences() {
                    return alliancePreferences;
                }

                /** 
                 * Set the 'AlliancePreferences' element value.
                 * 
                 * @param alliancePreferences
                 */
                public void setAlliancePreferences(
                        AlliancePreferencesType alliancePreferences) {
                    this.alliancePreferences = alliancePreferences;
                }

                /** 
                 * Get the 'FarePreferences' element value.
                 * 
                 * @return value
                 */
                public FarePreferencesType getFarePreferences() {
                    return farePreferences;
                }

                /** 
                 * Set the 'FarePreferences' element value.
                 * 
                 * @param farePreferences
                 */
                public void setFarePreferences(
                        FarePreferencesType farePreferences) {
                    this.farePreferences = farePreferences;
                }

                /** 
                 * Get the 'FlightPreferences' element value.
                 * 
                 * @return value
                 */
                public FlightPreferencesType getFlightPreferences() {
                    return flightPreferences;
                }

                /** 
                 * Set the 'FlightPreferences' element value.
                 * 
                 * @param flightPreferences
                 */
                public void setFlightPreferences(
                        FlightPreferencesType flightPreferences) {
                    this.flightPreferences = flightPreferences;
                }

                /** 
                 * Get the 'PricingMethodPreference' element value.
                 * 
                 * @return value
                 */
                public BestPricingPreferencesType getPricingMethodPreference() {
                    return pricingMethodPreference;
                }

                /** 
                 * Set the 'PricingMethodPreference' element value.
                 * 
                 * @param pricingMethodPreference
                 */
                public void setPricingMethodPreference(
                        BestPricingPreferencesType pricingMethodPreference) {
                    this.pricingMethodPreference = pricingMethodPreference;
                }

                /** 
                 * Get the 'SegMaxTimePreferences' element value.
                 * 
                 * @return value
                 */
                public FltSegmentMaxTimePreferencesType getSegMaxTimePreferences() {
                    return segMaxTimePreferences;
                }

                /** 
                 * Set the 'SegMaxTimePreferences' element value.
                 * 
                 * @param segMaxTimePreferences
                 */
                public void setSegMaxTimePreferences(
                        FltSegmentMaxTimePreferencesType segMaxTimePreferences) {
                    this.segMaxTimePreferences = segMaxTimePreferences;
                }

                /** 
                 * Get the 'ServicePricingOnlyPreference' element value.
                 * 
                 * @return value
                 */
                public ServicePricingOnlyPreference getServicePricingOnlyPreference() {
                    return servicePricingOnlyPreference;
                }

                /** 
                 * Set the 'ServicePricingOnlyPreference' element value.
                 * 
                 * @param servicePricingOnlyPreference
                 */
                public void setServicePricingOnlyPreference(
                        ServicePricingOnlyPreference servicePricingOnlyPreference) {
                    this.servicePricingOnlyPreference = servicePricingOnlyPreference;
                }

                /** 
                 * Get the 'TransferPreferences' element value.
                 * 
                 * @return value
                 */
                public TransferPreferencesType getTransferPreferences() {
                    return transferPreferences;
                }

                /** 
                 * Set the 'TransferPreferences' element value.
                 * 
                 * @param transferPreferences
                 */
                public void setTransferPreferences(
                        TransferPreferencesType transferPreferences) {
                    this.transferPreferences = transferPreferences;
                }

                /** 
                 * Get the 'CabinPreferences' element value.
                 * 
                 * @return value
                 */
                public CabinPreferencesType getCabinPreferences() {
                    return cabinPreferences;
                }

                /** 
                 * Set the 'CabinPreferences' element value.
                 * 
                 * @param cabinPreferences
                 */
                public void setCabinPreferences(
                        CabinPreferencesType cabinPreferences) {
                    this.cabinPreferences = cabinPreferences;
                }
            }
        }
        /** 
         * Function to request the removal of Order Items from this specific Order.

        If used alone (without a shopping query from the “Add” function), the RS will assume a cancellation is desired for the Order Items specified and will return incurred fees/refund amounts and/or alternative OfferItems.

        If used in conjunction with an “Add” function, the intention is to replace this deleted item with whatever has been requested in the Add function’s shopping query.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Delete" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element name="OrderItem" maxOccurs="unbounded">
         *         &lt;!-- Reference to inner class OrderItem -->
         *       &lt;/xs:element>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class Delete implements Serializable
        {
            private List<OrderItem> orderItemList = new ArrayList<OrderItem>();

            /** 
             * Get the list of 'OrderItem' element items.
             * 
             * @return list
             */
            public List<OrderItem> getOrderItemList() {
                return orderItemList;
            }

            /** 
             * Set the list of 'OrderItem' element items.
             * 
             * @param list
             */
            public void setOrderItemList(List<OrderItem> list) {
                orderItemList = list;
            }
            /** 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="OrderItem" maxOccurs="unbounded">
             *   &lt;xs:complexType>
             *     &lt;xs:sequence>
             *       &lt;xs:element type="xs:string" name="ServiceRetainRequestIDs" minOccurs="0"/>
             *     &lt;/xs:sequence>
             *     &lt;xs:attribute type="xs:string" use="required" name="OrderItemID"/>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * </pre>
             */
            public static class OrderItem implements Serializable
            {
                private String serviceRetainRequestIDs;
                private String orderItemID;

                /** 
                 * Get the 'ServiceRetainRequestIDs' element value. Reference to the Services within the specified Order Item which the Passenger would like the Airline to retain and return within the proposed Offer Item(s) in the OrderReshopRS.
                 * 
                 * @return value
                 */
                public String getServiceRetainRequestIDs() {
                    return serviceRetainRequestIDs;
                }

                /** 
                 * Set the 'ServiceRetainRequestIDs' element value. Reference to the Services within the specified Order Item which the Passenger would like the Airline to retain and return within the proposed Offer Item(s) in the OrderReshopRS.
                 * 
                 * @param serviceRetainRequestIDs
                 */
                public void setServiceRetainRequestIDs(
                        String serviceRetainRequestIDs) {
                    this.serviceRetainRequestIDs = serviceRetainRequestIDs;
                }

                /** 
                 * Get the 'OrderItemID' attribute value. Reference to the Order Item requested for deletion.
                 * 
                 * @return value
                 */
                public String getOrderItemID() {
                    return orderItemID;
                }

                /** 
                 * Set the 'OrderItemID' attribute value. Reference to the Order Item requested for deletion.
                 * 
                 * @param orderItemID
                 */
                public void setOrderItemID(String orderItemID) {
                    this.orderItemID = orderItemID;
                }
            }
        }
    }
    /** 
     * Function to reprice entire Order (if only OrderID is supplied within Query) or Specific OrderItems (if defined in nested OrderItem elements)
     * 
     * Schema fragment(s) for this class:
     * <pre>
     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Reprice">
     *   &lt;xs:complexType>
     *     &lt;xs:sequence>
     *       &lt;xs:element type="xs:string" name="OrderItemID" minOccurs="0" maxOccurs="unbounded"/>
     *     &lt;/xs:sequence>
     *   &lt;/xs:complexType>
     * &lt;/xs:element>
     * </pre>
     */
    public static class Reprice implements Serializable
    {
        private List<String> orderItemIDList = new ArrayList<String>();

        /** 
         * Get the list of 'OrderItemID' element items. Optionally specify which OrderItem needs to be repriced.
         * 
         * @return list
         */
        public List<String> getOrderItemIDList() {
            return orderItemIDList;
        }

        /** 
         * Set the list of 'OrderItemID' element items. Optionally specify which OrderItem needs to be repriced.
         * 
         * @param list
         */
        public void setOrderItemIDList(List<String> list) {
            orderItemIDList = list;
        }
    }
    /** 
     * Data Lists(s).
     * 
     * Schema fragment(s) for this class:
     * <pre>
     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="DataLists" minOccurs="0">
     *   &lt;xs:complexType>
     *     &lt;xs:sequence>
     *       &lt;xs:element name="PassengerList" minOccurs="0">
     *         &lt;!-- Reference to inner class PassengerList -->
     *       &lt;/xs:element>
     *       &lt;xs:element name="ContactList" minOccurs="0">
     *         &lt;!-- Reference to inner class ContactList -->
     *       &lt;/xs:element>
     *       &lt;xs:element ref="ns:BagDisclosureList" minOccurs="0"/>
     *       &lt;xs:element name="BaggageAllowanceList" minOccurs="0">
     *         &lt;!-- Reference to inner class BaggageAllowanceList -->
     *       &lt;/xs:element>
     *       &lt;xs:element ref="ns:ClassOfServiceList" minOccurs="0"/>
     *       &lt;xs:element ref="ns:ContentSourceList" minOccurs="0"/>
     *       &lt;xs:element name="DescriptionList" minOccurs="0">
     *         &lt;!-- Reference to inner class DescriptionList -->
     *       &lt;/xs:element>
     *       &lt;xs:element ref="ns:DisclosureList" minOccurs="0"/>
     *       &lt;xs:element ref="ns:FareList" minOccurs="0"/>
     *       &lt;xs:element name="FlightSegmentList" minOccurs="0">
     *         &lt;!-- Reference to inner class FlightSegmentList -->
     *       &lt;/xs:element>
     *       &lt;xs:element name="FlightList" minOccurs="0">
     *         &lt;!-- Reference to inner class FlightList -->
     *       &lt;/xs:element>
     *       &lt;xs:element name="OriginDestinationList" minOccurs="0">
     *         &lt;!-- Reference to inner class OriginDestinationList -->
     *       &lt;/xs:element>
     *       &lt;xs:element ref="ns:InstructionsList" minOccurs="0"/>
     *       &lt;xs:element ref="ns:MediaList" minOccurs="0"/>
     *       &lt;xs:element ref="ns:PenaltyList" minOccurs="0"/>
     *       &lt;xs:element name="PriceClassList" minOccurs="0">
     *         &lt;!-- Reference to inner class PriceClassList -->
     *       &lt;/xs:element>
     *       &lt;xs:element name="ServiceDefinitionList" minOccurs="0">
     *         &lt;!-- Reference to inner class ServiceDefinitionList -->
     *       &lt;/xs:element>
     *       &lt;xs:element ref="ns:TermsList" minOccurs="0"/>
     *       &lt;xs:element name="SeatDefinitionList" minOccurs="0">
     *         &lt;!-- Reference to inner class SeatDefinitionList -->
     *       &lt;/xs:element>
     *     &lt;/xs:sequence>
     *   &lt;/xs:complexType>
     * &lt;/xs:element>
     * </pre>
     */
    public static class DataLists implements Serializable
    {
        private PassengerList passengerList;
        private ContactList contactList;
        private ListOfBagDisclosureType bagDisclosureList;
        private BaggageAllowanceList baggageAllowanceList;
        private ListOfClassOfServiceType classOfServiceList;
        private ListOfContentSourceType contentSourceList;
        private DescriptionList descriptionList;
        private ListOfDisclosureType disclosureList;
        private FareList fareList;
        private FlightSegmentList flightSegmentList;
        private FlightList flightList;
        private OriginDestinationList originDestinationList;
        private ListOfOfferInstructionsType instructionsList;
        private ListOfMediaType mediaList;
        private ListOfOfferPenaltyType penaltyList;
        private PriceClassList priceClassList;
        private ServiceDefinitionList serviceDefinitionList;
        private ListOfOfferTermsType termsList;
        private SeatDefinitionList seatDefinitionList;

        /** 
         * Get the 'PassengerList' element value.
         * 
         * @return value
         */
        public PassengerList getPassengerList() {
            return passengerList;
        }

        /** 
         * Set the 'PassengerList' element value.
         * 
         * @param passengerList
         */
        public void setPassengerList(PassengerList passengerList) {
            this.passengerList = passengerList;
        }

        /** 
         * Get the 'ContactList' element value.
         * 
         * @return value
         */
        public ContactList getContactList() {
            return contactList;
        }

        /** 
         * Set the 'ContactList' element value.
         * 
         * @param contactList
         */
        public void setContactList(ContactList contactList) {
            this.contactList = contactList;
        }

        /** 
         * Get the 'BagDisclosureList' element value.
         * 
         * @return value
         */
        public ListOfBagDisclosureType getBagDisclosureList() {
            return bagDisclosureList;
        }

        /** 
         * Set the 'BagDisclosureList' element value.
         * 
         * @param bagDisclosureList
         */
        public void setBagDisclosureList(
                ListOfBagDisclosureType bagDisclosureList) {
            this.bagDisclosureList = bagDisclosureList;
        }

        /** 
         * Get the 'BaggageAllowanceList' element value.
         * 
         * @return value
         */
        public BaggageAllowanceList getBaggageAllowanceList() {
            return baggageAllowanceList;
        }

        /** 
         * Set the 'BaggageAllowanceList' element value.
         * 
         * @param baggageAllowanceList
         */
        public void setBaggageAllowanceList(
                BaggageAllowanceList baggageAllowanceList) {
            this.baggageAllowanceList = baggageAllowanceList;
        }

        /** 
         * Get the 'ClassOfServiceList' element value.
         * 
         * @return value
         */
        public ListOfClassOfServiceType getClassOfServiceList() {
            return classOfServiceList;
        }

        /** 
         * Set the 'ClassOfServiceList' element value.
         * 
         * @param classOfServiceList
         */
        public void setClassOfServiceList(
                ListOfClassOfServiceType classOfServiceList) {
            this.classOfServiceList = classOfServiceList;
        }

        /** 
         * Get the 'ContentSourceList' element value.
         * 
         * @return value
         */
        public ListOfContentSourceType getContentSourceList() {
            return contentSourceList;
        }

        /** 
         * Set the 'ContentSourceList' element value.
         * 
         * @param contentSourceList
         */
        public void setContentSourceList(
                ListOfContentSourceType contentSourceList) {
            this.contentSourceList = contentSourceList;
        }

        /** 
         * Get the 'DescriptionList' element value.
         * 
         * @return value
         */
        public DescriptionList getDescriptionList() {
            return descriptionList;
        }

        /** 
         * Set the 'DescriptionList' element value.
         * 
         * @param descriptionList
         */
        public void setDescriptionList(DescriptionList descriptionList) {
            this.descriptionList = descriptionList;
        }

        /** 
         * Get the 'DisclosureList' element value.
         * 
         * @return value
         */
        public ListOfDisclosureType getDisclosureList() {
            return disclosureList;
        }

        /** 
         * Set the 'DisclosureList' element value.
         * 
         * @param disclosureList
         */
        public void setDisclosureList(ListOfDisclosureType disclosureList) {
            this.disclosureList = disclosureList;
        }

        /** 
         * Get the 'FareList' element value.
         * 
         * @return value
         */
        public FareList getFareList() {
            return fareList;
        }

        /** 
         * Set the 'FareList' element value.
         * 
         * @param fareList
         */
        public void setFareList(FareList fareList) {
            this.fareList = fareList;
        }

        /** 
         * Get the 'FlightSegmentList' element value.
         * 
         * @return value
         */
        public FlightSegmentList getFlightSegmentList() {
            return flightSegmentList;
        }

        /** 
         * Set the 'FlightSegmentList' element value.
         * 
         * @param flightSegmentList
         */
        public void setFlightSegmentList(FlightSegmentList flightSegmentList) {
            this.flightSegmentList = flightSegmentList;
        }

        /** 
         * Get the 'FlightList' element value.
         * 
         * @return value
         */
        public FlightList getFlightList() {
            return flightList;
        }

        /** 
         * Set the 'FlightList' element value.
         * 
         * @param flightList
         */
        public void setFlightList(FlightList flightList) {
            this.flightList = flightList;
        }

        /** 
         * Get the 'OriginDestinationList' element value.
         * 
         * @return value
         */
        public OriginDestinationList getOriginDestinationList() {
            return originDestinationList;
        }

        /** 
         * Set the 'OriginDestinationList' element value.
         * 
         * @param originDestinationList
         */
        public void setOriginDestinationList(
                OriginDestinationList originDestinationList) {
            this.originDestinationList = originDestinationList;
        }

        /** 
         * Get the 'InstructionsList' element value.
         * 
         * @return value
         */
        public ListOfOfferInstructionsType getInstructionsList() {
            return instructionsList;
        }

        /** 
         * Set the 'InstructionsList' element value.
         * 
         * @param instructionsList
         */
        public void setInstructionsList(
                ListOfOfferInstructionsType instructionsList) {
            this.instructionsList = instructionsList;
        }

        /** 
         * Get the 'MediaList' element value.
         * 
         * @return value
         */
        public ListOfMediaType getMediaList() {
            return mediaList;
        }

        /** 
         * Set the 'MediaList' element value.
         * 
         * @param mediaList
         */
        public void setMediaList(ListOfMediaType mediaList) {
            this.mediaList = mediaList;
        }

        /** 
         * Get the 'PenaltyList' element value.
         * 
         * @return value
         */
        public ListOfOfferPenaltyType getPenaltyList() {
            return penaltyList;
        }

        /** 
         * Set the 'PenaltyList' element value.
         * 
         * @param penaltyList
         */
        public void setPenaltyList(ListOfOfferPenaltyType penaltyList) {
            this.penaltyList = penaltyList;
        }

        /** 
         * Get the 'PriceClassList' element value.
         * 
         * @return value
         */
        public PriceClassList getPriceClassList() {
            return priceClassList;
        }

        /** 
         * Set the 'PriceClassList' element value.
         * 
         * @param priceClassList
         */
        public void setPriceClassList(PriceClassList priceClassList) {
            this.priceClassList = priceClassList;
        }

        /** 
         * Get the 'ServiceDefinitionList' element value.
         * 
         * @return value
         */
        public ServiceDefinitionList getServiceDefinitionList() {
            return serviceDefinitionList;
        }

        /** 
         * Set the 'ServiceDefinitionList' element value.
         * 
         * @param serviceDefinitionList
         */
        public void setServiceDefinitionList(
                ServiceDefinitionList serviceDefinitionList) {
            this.serviceDefinitionList = serviceDefinitionList;
        }

        /** 
         * Get the 'TermsList' element value.
         * 
         * @return value
         */
        public ListOfOfferTermsType getTermsList() {
            return termsList;
        }

        /** 
         * Set the 'TermsList' element value.
         * 
         * @param termsList
         */
        public void setTermsList(ListOfOfferTermsType termsList) {
            this.termsList = termsList;
        }

        /** 
         * Get the 'SeatDefinitionList' element value.
         * 
         * @return value
         */
        public SeatDefinitionList getSeatDefinitionList() {
            return seatDefinitionList;
        }

        /** 
         * Set the 'SeatDefinitionList' element value.
         * 
         * @param seatDefinitionList
         */
        public void setSeatDefinitionList(SeatDefinitionList seatDefinitionList) {
            this.seatDefinitionList = seatDefinitionList;
        }
        /** 
         * Passenger data list definition.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="PassengerList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element type="ns:PassengerType" name="Passenger" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class PassengerList implements Serializable
        {
            private List<PassengerType> passengerList = new ArrayList<PassengerType>();

            /** 
             * Get the list of 'Passenger' element items. Any person except members of the crew carried or to be carried with the consent of the carrier, on board of any transport vehicle such as aircraft, train, bus, ship.
             * 
             * @return list
             */
            public List<PassengerType> getPassengerList() {
                return passengerList;
            }

            /** 
             * Set the list of 'Passenger' element items. Any person except members of the crew carried or to be carried with the consent of the carrier, on board of any transport vehicle such as aircraft, train, bus, ship.
             * 
             * @param list
             */
            public void setPassengerList(List<PassengerType> list) {
                passengerList = list;
            }
        }
        /** 
         * Contact data list definition.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="ContactList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element type="ns:ContactInformationType" name="ContactInformation" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class ContactList implements Serializable
        {
            private List<ContactInformationType> contactInformationList = new ArrayList<ContactInformationType>();

            /** 
             * Get the list of 'ContactInformation' element items. The electronic or geographic address which a party has provided as the contact channel. For example, contact email, contact postal address, contact phone number.
             * 
             * @return list
             */
            public List<ContactInformationType> getContactInformationList() {
                return contactInformationList;
            }

            /** 
             * Set the list of 'ContactInformation' element items. The electronic or geographic address which a party has provided as the contact channel. For example, contact email, contact postal address, contact phone number.
             * 
             * @param list
             */
            public void setContactInformationList(
                    List<ContactInformationType> list) {
                contactInformationList = list;
            }
        }
        /** 
         * Checked or Carry-On Baggage Allowance detail. Use BaggageCategory to specify which of the two.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="BaggageAllowanceList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element name="BaggageAllowance" maxOccurs="unbounded">
         *         &lt;!-- Reference to inner class BaggageAllowance -->
         *       &lt;/xs:element>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class BaggageAllowanceList implements Serializable
        {
            private List<BaggageAllowance> baggageAllowanceList = new ArrayList<BaggageAllowance>();

            /** 
             * Get the list of 'BaggageAllowance' element items.
             * 
             * @return list
             */
            public List<BaggageAllowance> getBaggageAllowanceList() {
                return baggageAllowanceList;
            }

            /** 
             * Set the list of 'BaggageAllowance' element items.
             * 
             * @param list
             */
            public void setBaggageAllowanceList(List<BaggageAllowance> list) {
                baggageAllowanceList = list;
            }
            /** 
             * Checked or Carry-On Baggage Allowance detail. Use BaggageCategory to specify which of the two.
             * 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="BaggageAllowance" maxOccurs="unbounded">
             *   &lt;xs:complexType>
             *     &lt;xs:sequence>
             *       &lt;xs:element type="ns:BaggageCategoryListType" name="BaggageCategory"/>
             *       &lt;xs:element ref="ns:AllowanceDescription" minOccurs="0"/>
             *       &lt;xs:element ref="ns:DimensionAllowance" minOccurs="0"/>
             *       &lt;xs:element ref="ns:PieceAllowance" minOccurs="0" maxOccurs="unbounded"/>
             *       &lt;xs:element ref="ns:WeightAllowance" minOccurs="0"/>
             *       &lt;xs:element type="ns:BaggageDeterminingCarrierType" name="BaggageDeterminingCarrier" minOccurs="0"/>
             *     &lt;/xs:sequence>
             *     &lt;xs:attribute type="xs:string" use="required" name="BaggageAllowanceID"/>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * </pre>
             */
            public static class BaggageAllowance implements Serializable
            {
                private BaggageCategoryListType baggageCategory;
                private BagAllowanceDescType allowanceDescription;
                private BagAllowanceDimensionType dimensionAllowance;
                private List<BagAllowancePieceType> pieceAllowanceList = new ArrayList<BagAllowancePieceType>();
                private WeightAllowance weightAllowance;
                private BaggageDeterminingCarrierType baggageDeterminingCarrier;
                private String baggageAllowanceID;

                /** 
                 * Get the 'BaggageCategory' element value. Enumeration:

                CarryOn
                Checked
                 * 
                 * @return value
                 */
                public BaggageCategoryListType getBaggageCategory() {
                    return baggageCategory;
                }

                /** 
                 * Set the 'BaggageCategory' element value. Enumeration:

                CarryOn
                Checked
                 * 
                 * @param baggageCategory
                 */
                public void setBaggageCategory(
                        BaggageCategoryListType baggageCategory) {
                    this.baggageCategory = baggageCategory;
                }

                /** 
                 * Get the 'AllowanceDescription' element value.
                 * 
                 * @return value
                 */
                public BagAllowanceDescType getAllowanceDescription() {
                    return allowanceDescription;
                }

                /** 
                 * Set the 'AllowanceDescription' element value.
                 * 
                 * @param allowanceDescription
                 */
                public void setAllowanceDescription(
                        BagAllowanceDescType allowanceDescription) {
                    this.allowanceDescription = allowanceDescription;
                }

                /** 
                 * Get the 'DimensionAllowance' element value.
                 * 
                 * @return value
                 */
                public BagAllowanceDimensionType getDimensionAllowance() {
                    return dimensionAllowance;
                }

                /** 
                 * Set the 'DimensionAllowance' element value.
                 * 
                 * @param dimensionAllowance
                 */
                public void setDimensionAllowance(
                        BagAllowanceDimensionType dimensionAllowance) {
                    this.dimensionAllowance = dimensionAllowance;
                }

                /** 
                 * Get the list of 'PieceAllowance' element items.
                 * 
                 * @return list
                 */
                public List<BagAllowancePieceType> getPieceAllowanceList() {
                    return pieceAllowanceList;
                }

                /** 
                 * Set the list of 'PieceAllowance' element items.
                 * 
                 * @param list
                 */
                public void setPieceAllowanceList(
                        List<BagAllowancePieceType> list) {
                    pieceAllowanceList = list;
                }

                /** 
                 * Get the 'WeightAllowance' element value.
                 * 
                 * @return value
                 */
                public WeightAllowance getWeightAllowance() {
                    return weightAllowance;
                }

                /** 
                 * Set the 'WeightAllowance' element value.
                 * 
                 * @param weightAllowance
                 */
                public void setWeightAllowance(WeightAllowance weightAllowance) {
                    this.weightAllowance = weightAllowance;
                }

                /** 
                 * Get the 'BaggageDeterminingCarrier' element value. Baggage Determining Carrier as defined in resolution 722 (f), and who’s baggage rules apply.
                 * 
                 * @return value
                 */
                public BaggageDeterminingCarrierType getBaggageDeterminingCarrier() {
                    return baggageDeterminingCarrier;
                }

                /** 
                 * Set the 'BaggageDeterminingCarrier' element value. Baggage Determining Carrier as defined in resolution 722 (f), and who’s baggage rules apply.
                 * 
                 * @param baggageDeterminingCarrier
                 */
                public void setBaggageDeterminingCarrier(
                        BaggageDeterminingCarrierType baggageDeterminingCarrier) {
                    this.baggageDeterminingCarrier = baggageDeterminingCarrier;
                }

                /** 
                 * Get the 'BaggageAllowanceID' attribute value.
                 * 
                 * @return value
                 */
                public String getBaggageAllowanceID() {
                    return baggageAllowanceID;
                }

                /** 
                 * Set the 'BaggageAllowanceID' attribute value.
                 * 
                 * @param baggageAllowanceID
                 */
                public void setBaggageAllowanceID(String baggageAllowanceID) {
                    this.baggageAllowanceID = baggageAllowanceID;
                }
            }
        }
        /** 
         * Description Details
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="DescriptionList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element ref="ns:Description" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class DescriptionList implements Serializable
        {
            private List<com.gridnine.xtrip.server.gds.tandc.wsmodel.Description> descriptionList = new ArrayList<com.gridnine.xtrip.server.gds.tandc.wsmodel.Description>();

            /** 
             * Get the list of 'Description' element items.
             * 
             * @return list
             */
            public List<com.gridnine.xtrip.server.gds.tandc.wsmodel.Description> getDescriptionList() {
                return descriptionList;
            }

            /** 
             * Set the list of 'Description' element items.
             * 
             * @param list
             */
            public void setDescriptionList(
                    List<com.gridnine.xtrip.server.gds.tandc.wsmodel.Description> list) {
                descriptionList = list;
            }
        }
        /** 
         * List of Flight Segments
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="FlightSegmentList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element type="ns:ListOfFlightSegmentType" name="FlightSegment" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class FlightSegmentList implements Serializable
        {
            private List<ListOfFlightSegmentType> flightSegmentList = new ArrayList<ListOfFlightSegmentType>();

            /** 
             * Get the list of 'FlightSegment' element items.
             * 
             * @return list
             */
            public List<ListOfFlightSegmentType> getFlightSegmentList() {
                return flightSegmentList;
            }

            /** 
             * Set the list of 'FlightSegment' element items.
             * 
             * @param list
             */
            public void setFlightSegmentList(List<ListOfFlightSegmentType> list) {
                flightSegmentList = list;
            }
        }
        /** 
         * List of Flight elements.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="FlightList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element name="Flight" maxOccurs="unbounded">
         *         &lt;!-- Reference to inner class Flight -->
         *       &lt;/xs:element>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class FlightList implements Serializable
        {
            private List<Flight> flightList = new ArrayList<Flight>();

            /** 
             * Get the list of 'Flight' element items.
             * 
             * @return list
             */
            public List<Flight> getFlightList() {
                return flightList;
            }

            /** 
             * Set the list of 'Flight' element items.
             * 
             * @param list
             */
            public void setFlightList(List<Flight> list) {
                flightList = list;
            }
            /** 
             * Journey information describing the travel between an airport/city where travel commences and an airport/city where travel ultimately terminates. A journey may be comprised of one or more segments. Note: This node will be renamed to journey in future releases.
             * 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Flight" maxOccurs="unbounded">
             *   &lt;xs:complexType>
             *     &lt;xs:sequence>
             *       &lt;xs:element type="ns:TotalJourneyType" name="Journey" minOccurs="0"/>
             *       &lt;xs:element ref="ns:SegmentReferences"/>
             *       &lt;xs:element name="Settlement" minOccurs="0">
             *         &lt;!-- Reference to inner class Settlement -->
             *       &lt;/xs:element>
             *     &lt;/xs:sequence>
             *     &lt;xs:attributeGroup ref="ns:ObjAssociationAttrGroup"/>
             *     &lt;xs:attribute type="xs:string" name="FlightKey"/>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * 
             * &lt;xs:attributeGroup xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="ObjAssociationAttrGroup">
             *   &lt;xs:attribute type="xs:string" use="optional" name="refs"/>
             * &lt;/xs:attributeGroup>
             * </pre>
             */
            public static class Flight implements Serializable
            {
                private TotalJourneyType journey;
                private SegmentReferences segmentReferences;
                private Settlement settlement;
                private String refs;
                private String flightKey;

                /** 
                 * Get the 'Journey' element value. Flight journey information.
                 * 
                 * @return value
                 */
                public TotalJourneyType getJourney() {
                    return journey;
                }

                /** 
                 * Set the 'Journey' element value. Flight journey information.
                 * 
                 * @param journey
                 */
                public void setJourney(TotalJourneyType journey) {
                    this.journey = journey;
                }

                /** 
                 * Get the 'SegmentReferences' element value.
                 * 
                 * @return value
                 */
                public SegmentReferences getSegmentReferences() {
                    return segmentReferences;
                }

                /** 
                 * Set the 'SegmentReferences' element value.
                 * 
                 * @param segmentReferences
                 */
                public void setSegmentReferences(
                        SegmentReferences segmentReferences) {
                    this.segmentReferences = segmentReferences;
                }

                /** 
                 * Get the 'Settlement' element value.
                 * 
                 * @return value
                 */
                public Settlement getSettlement() {
                    return settlement;
                }

                /** 
                 * Set the 'Settlement' element value.
                 * 
                 * @param settlement
                 */
                public void setSettlement(Settlement settlement) {
                    this.settlement = settlement;
                }

                /** 
                 * Get the 'refs' attribute value. Object association(s) between this object and any other object or metadata instance(s).
                 * 
                 * @return value
                 */
                public String getRefs() {
                    return refs;
                }

                /** 
                 * Set the 'refs' attribute value. Object association(s) between this object and any other object or metadata instance(s).
                 * 
                 * @param refs
                 */
                public void setRefs(String refs) {
                    this.refs = refs;
                }

                /** 
                 * Get the 'FlightKey' attribute value. A globally unique identifier for this Flight instance. Example: flight1
                 * 
                 * @return value
                 */
                public String getFlightKey() {
                    return flightKey;
                }

                /** 
                 * Set the 'FlightKey' attribute value. A globally unique identifier for this Flight instance. Example: flight1
                 * 
                 * @param flightKey
                 */
                public void setFlightKey(String flightKey) {
                    this.flightKey = flightKey;
                }
                /** 
                 * Interline Settlement information between airlines, including method of settlement, interline settlement value and currency.
                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Settlement" minOccurs="0">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:element type="xs:string" name="Method"/>
                 *       &lt;xs:element type="ns:CurrencyAmountOptType" name="InterlineSettlementValue" minOccurs="0"/>
                 *     &lt;/xs:sequence>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class Settlement implements Serializable
                {
                    private String method;
                    private CurrencyAmountOptType interlineSettlementValue;

                    /** 
                     * Get the 'Method' element value. Service settlement method.
                    Examples:
                    AD (Internal Airline Document)
                    DS (Direct Settlement)
                    EA (EMD Associated or Internal Airline Document)
                    ES (EMD Standalone )
                    FI (Fare Integrated)
                    MA (Flight Associated, Settled via MCO)
                    MS (Standalone, Settled via MCO)

                    Encoding Scheme: IATA PADIS Codeset SST (Service Settlement Type)
                     * 
                     * @return value
                     */
                    public String getMethod() {
                        return method;
                    }

                    /** 
                     * Set the 'Method' element value. Service settlement method.
                    Examples:
                    AD (Internal Airline Document)
                    DS (Direct Settlement)
                    EA (EMD Associated or Internal Airline Document)
                    ES (EMD Standalone )
                    FI (Fare Integrated)
                    MA (Flight Associated, Settled via MCO)
                    MS (Standalone, Settled via MCO)

                    Encoding Scheme: IATA PADIS Codeset SST (Service Settlement Type)
                     * 
                     * @param method
                     */
                    public void setMethod(String method) {
                        this.method = method;
                    }

                    /** 
                     * Get the 'InterlineSettlementValue' element value. Settlement (Price) Value. Example: 300 (USD)

                    Note: This represents a Settlement value that is only used between the POA and ORA.
                     * 
                     * @return value
                     */
                    public CurrencyAmountOptType getInterlineSettlementValue() {
                        return interlineSettlementValue;
                    }

                    /** 
                     * Set the 'InterlineSettlementValue' element value. Settlement (Price) Value. Example: 300 (USD)

                    Note: This represents a Settlement value that is only used between the POA and ORA.
                     * 
                     * @param interlineSettlementValue
                     */
                    public void setInterlineSettlementValue(
                            CurrencyAmountOptType interlineSettlementValue) {
                        this.interlineSettlementValue = interlineSettlementValue;
                    }
                }
            }
        }
        /** 
         * List of Origin/Destination elements.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="OriginDestinationList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element ref="ns:OriginDestination" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class OriginDestinationList implements Serializable
        {
            private List<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination> originDestinationList = new ArrayList<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination>();

            /** 
             * Get the list of 'OriginDestination' element items.
             * 
             * @return list
             */
            public List<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination> getOriginDestinationList() {
                return originDestinationList;
            }

            /** 
             * Set the list of 'OriginDestination' element items.
             * 
             * @param list
             */
            public void setOriginDestinationList(
                    List<com.gridnine.xtrip.server.gds.tandc.wsmodel.OriginDestination> list) {
                originDestinationList = list;
            }
        }
        /** 
         * Price Class list.
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="PriceClassList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element type="ns:PriceClassType" name="PriceClass" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class PriceClassList implements Serializable
        {
            private List<PriceClassType> priceClassList = new ArrayList<PriceClassType>();

            /** 
             * Get the list of 'PriceClass' element items.
             * 
             * @return list
             */
            public List<PriceClassType> getPriceClassList() {
                return priceClassList;
            }

            /** 
             * Set the list of 'PriceClass' element items.
             * 
             * @param list
             */
            public void setPriceClassList(List<PriceClassType> list) {
                priceClassList = list;
            }
        }
        /** 
         * Description and characteristics of a Service
         * 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="ServiceDefinitionList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element type="ns:ServiceDefinitionType" name="ServiceDefinition" maxOccurs="unbounded"/>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class ServiceDefinitionList implements Serializable
        {
            private List<ServiceDefinitionType> serviceDefinitionList = new ArrayList<ServiceDefinitionType>();

            /** 
             * Get the list of 'ServiceDefinition' element items.
             * 
             * @return list
             */
            public List<ServiceDefinitionType> getServiceDefinitionList() {
                return serviceDefinitionList;
            }

            /** 
             * Set the list of 'ServiceDefinition' element items.
             * 
             * @param list
             */
            public void setServiceDefinitionList(
                    List<ServiceDefinitionType> list) {
                serviceDefinitionList = list;
            }
        }
        /** 
         * Schema fragment(s) for this class:
         * <pre>
         * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="SeatDefinitionList" minOccurs="0">
         *   &lt;xs:complexType>
         *     &lt;xs:sequence>
         *       &lt;xs:element name="SeatDefinition" minOccurs="0" maxOccurs="unbounded">
         *         &lt;!-- Reference to inner class SeatDefinition -->
         *       &lt;/xs:element>
         *     &lt;/xs:sequence>
         *   &lt;/xs:complexType>
         * &lt;/xs:element>
         * </pre>
         */
        public static class SeatDefinitionList implements Serializable
        {
            private List<SeatDefinition> seatDefinitionList = new ArrayList<SeatDefinition>();

            /** 
             * Get the list of 'SeatDefinition' element items.
             * 
             * @return list
             */
            public List<SeatDefinition> getSeatDefinitionList() {
                return seatDefinitionList;
            }

            /** 
             * Set the list of 'SeatDefinition' element items.
             * 
             * @param list
             */
            public void setSeatDefinitionList(List<SeatDefinition> list) {
                seatDefinitionList = list;
            }
            /** 
             * Seat information by aircraft section.
             * 
             * Schema fragment(s) for this class:
             * <pre>
             * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="SeatDefinition" minOccurs="0" maxOccurs="unbounded">
             *   &lt;xs:complexType>
             *     &lt;xs:sequence>
             *       &lt;xs:element name="Description" minOccurs="0" maxOccurs="unbounded">
             *         &lt;!-- Reference to inner class Description -->
             *       &lt;/xs:element>
             *       &lt;xs:element type="ns:SeatMapMessageType" name="Marketing" minOccurs="0"/>
             *       &lt;xs:element type="xs:string" name="SeatCharacteristicCode" minOccurs="0" maxOccurs="unbounded"/>
             *       &lt;xs:element type="ns:SizeUnitSimpleType" name="UOM" minOccurs="0"/>
             *       &lt;xs:element type="xs:decimal" name="SeatWidthLow" minOccurs="0"/>
             *       &lt;xs:element type="xs:decimal" name="SeatPitchLow" minOccurs="0"/>
             *       &lt;xs:element name="Keywords" minOccurs="0">
             *         &lt;!-- Reference to inner class Keywords -->
             *       &lt;/xs:element>
             *     &lt;/xs:sequence>
             *     &lt;xs:attribute type="xs:string" use="required" name="SeatDefinitionID"/>
             *   &lt;/xs:complexType>
             * &lt;/xs:element>
             * </pre>
             */
            public static class SeatDefinition implements Serializable
            {
                private List<Description> descriptionList = new ArrayList<Description>();
                private SeatMapMessageType marketing;
                private List<String> seatCharacteristicCodeList = new ArrayList<String>();
                private SizeUnitSimpleType UOM;
                private BigDecimal seatWidthLow;
                private BigDecimal seatPitchLow;
                private Keywords keywords;
                private String seatDefinitionID;

                /** 
                 * Get the list of 'Description' element items.
                 * 
                 * @return list
                 */
                public List<Description> getDescriptionList() {
                    return descriptionList;
                }

                /** 
                 * Set the list of 'Description' element items.
                 * 
                 * @param list
                 */
                public void setDescriptionList(List<Description> list) {
                    descriptionList = list;
                }

                /** 
                 * Get the 'Marketing' element value. Images and/or links for the entire section defined by the range of rows.
                 * 
                 * @return value
                 */
                public SeatMapMessageType getMarketing() {
                    return marketing;
                }

                /** 
                 * Set the 'Marketing' element value. Images and/or links for the entire section defined by the range of rows.
                 * 
                 * @param marketing
                 */
                public void setMarketing(SeatMapMessageType marketing) {
                    this.marketing = marketing;
                }

                /** 
                 * Get the list of 'SeatCharacteristicCode' element items. Refers to aircraft seat characteristics; seat option codes.  Examples: E (EXIT ROW), M (Middle Seat), A (Aisle seat), L (Left side of aircraft), Q (Bulk Head).  Refer to PADIS code list 9825.
                 * 
                 * @return list
                 */
                public List<String> getSeatCharacteristicCodeList() {
                    return seatCharacteristicCodeList;
                }

                /** 
                 * Set the list of 'SeatCharacteristicCode' element items. Refers to aircraft seat characteristics; seat option codes.  Examples: E (EXIT ROW), M (Middle Seat), A (Aisle seat), L (Left side of aircraft), Q (Bulk Head).  Refer to PADIS code list 9825.
                 * 
                 * @param list
                 */
                public void setSeatCharacteristicCodeList(List<String> list) {
                    seatCharacteristicCodeList = list;
                }

                /** 
                 * Get the 'UOM' element value. Associated Keyword Unit of Measure, if relevant. Example: INCH
                 * 
                 * @return value
                 */
                public SizeUnitSimpleType getUOM() {
                    return UOM;
                }

                /** 
                 * Set the 'UOM' element value. Associated Keyword Unit of Measure, if relevant. Example: INCH
                 * 
                 * @param UOM
                 */
                public void setUOM(SizeUnitSimpleType UOM) {
                    this.UOM = UOM;
                }

                /** 
                 * Get the 'SeatWidthLow' element value. Lowest Width Measurement value for this section. Example: 20
                 * 
                 * @return value
                 */
                public BigDecimal getSeatWidthLow() {
                    return seatWidthLow;
                }

                /** 
                 * Set the 'SeatWidthLow' element value. Lowest Width Measurement value for this section. Example: 20
                 * 
                 * @param seatWidthLow
                 */
                public void setSeatWidthLow(BigDecimal seatWidthLow) {
                    this.seatWidthLow = seatWidthLow;
                }

                /** 
                 * Get the 'SeatPitchLow' element value. Lowest Pitch Measurement value for this section. Example: 32
                 * 
                 * @return value
                 */
                public BigDecimal getSeatPitchLow() {
                    return seatPitchLow;
                }

                /** 
                 * Set the 'SeatPitchLow' element value. Lowest Pitch Measurement value for this section. Example: 32
                 * 
                 * @param seatPitchLow
                 */
                public void setSeatPitchLow(BigDecimal seatPitchLow) {
                    this.seatPitchLow = seatPitchLow;
                }

                /** 
                 * Get the 'Keywords' element value.
                 * 
                 * @return value
                 */
                public Keywords getKeywords() {
                    return keywords;
                }

                /** 
                 * Set the 'Keywords' element value.
                 * 
                 * @param keywords
                 */
                public void setKeywords(Keywords keywords) {
                    this.keywords = keywords;
                }

                /** 
                 * Get the 'SeatDefinitionID' attribute value.
                 * 
                 * @return value
                 */
                public String getSeatDefinitionID() {
                    return seatDefinitionID;
                }

                /** 
                 * Set the 'SeatDefinitionID' attribute value.
                 * 
                 * @param seatDefinitionID
                 */
                public void setSeatDefinitionID(String seatDefinitionID) {
                    this.seatDefinitionID = seatDefinitionID;
                }
                /** 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Description" minOccurs="0" maxOccurs="unbounded">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:sequence>
                 *         &lt;xs:element name="Text" minOccurs="0">
                 *           &lt;!-- Reference to inner class Text -->
                 *         &lt;/xs:element>
                 *         &lt;xs:element type="xs:string" name="MarkupStyle" minOccurs="0"/>
                 *       &lt;/xs:sequence>
                 *       &lt;xs:element type="xs:string" name="Link" minOccurs="0"/>
                 *       &lt;xs:element name="Media" minOccurs="0" maxOccurs="unbounded">
                 *         &lt;!-- Reference to inner class Media -->
                 *       &lt;/xs:element>
                 *     &lt;/xs:sequence>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class Description implements Serializable
                {
                    private Text text;
                    private String markupStyle;
                    private String link;
                    private List<Media> mediaList = new ArrayList<Media>();

                    /** 
                     * Get the 'Text' element value.
                     * 
                     * @return value
                     */
                    public Text getText() {
                        return text;
                    }

                    /** 
                     * Set the 'Text' element value.
                     * 
                     * @param text
                     */
                    public void setText(Text text) {
                        this.text = text;
                    }

                    /** 
                     * Get the 'MarkupStyle' element value. Mark-up Style. Example: HTML
                     * 
                     * @return value
                     */
                    public String getMarkupStyle() {
                        return markupStyle;
                    }

                    /** 
                     * Set the 'MarkupStyle' element value. Mark-up Style. Example: HTML
                     * 
                     * @param markupStyle
                     */
                    public void setMarkupStyle(String markupStyle) {
                        this.markupStyle = markupStyle;
                    }

                    /** 
                     * Get the 'Link' element value. LINK to Description information source.
                     * 
                     * @return value
                     */
                    public String getLink() {
                        return link;
                    }

                    /** 
                     * Set the 'Link' element value. LINK to Description information source.
                     * 
                     * @param link
                     */
                    public void setLink(String link) {
                        this.link = link;
                    }

                    /** 
                     * Get the list of 'Media' element items.
                     * 
                     * @return list
                     */
                    public List<Media> getMediaList() {
                        return mediaList;
                    }

                    /** 
                     * Set the list of 'Media' element items.
                     * 
                     * @param list
                     */
                    public void setMediaList(List<Media> list) {
                        mediaList = list;
                    }
                    /** 
                     * Description value.
                     * 
                     * Schema fragment(s) for this class:
                     * <pre>
                     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Text" minOccurs="0">
                     *   &lt;xs:complexType>
                     *     &lt;xs:simpleContent>
                     *       &lt;xs:extension base="xs:string">
                     *         &lt;xs:attributeGroup ref="ns:ObjAssociationAttrGroup"/>
                     *       &lt;/xs:extension>
                     *     &lt;/xs:simpleContent>
                     *   &lt;/xs:complexType>
                     * &lt;/xs:element>
                     * 
                     * &lt;xs:attributeGroup xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="ObjAssociationAttrGroup">
                     *   &lt;xs:attribute type="xs:string" use="optional" name="refs"/>
                     * &lt;/xs:attributeGroup>
                     * </pre>
                     */
                    public static class Text implements Serializable
                    {
                        private String string;
                        private String refs;

                        /** 
                         * Get the extension value.
                         * 
                         * @return value
                         */
                        public String getString() {
                            return string;
                        }

                        /** 
                         * Set the extension value.
                         * 
                         * @param string
                         */
                        public void setString(String string) {
                            this.string = string;
                        }

                        /** 
                         * Get the 'refs' attribute value. Object association(s) between this object and any other object or metadata instance(s).
                         * 
                         * @return value
                         */
                        public String getRefs() {
                            return refs;
                        }

                        /** 
                         * Set the 'refs' attribute value. Object association(s) between this object and any other object or metadata instance(s).
                         * 
                         * @param refs
                         */
                        public void setRefs(String refs) {
                            this.refs = refs;
                        }
                    }
                    /** 
                     * Description-associated Media Object(s).
                     * 
                     * Schema fragment(s) for this class:
                     * <pre>
                     * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Media" minOccurs="0" maxOccurs="unbounded">
                     *   &lt;xs:complexType>
                     *     &lt;xs:choice>
                     *       &lt;xs:element type="ns:UniqueIDContextType" name="ObjectID"/>
                     *       &lt;xs:element ref="ns:MediaLink"/>
                     *       &lt;xs:element type="ns:UniqueIDContextType" name="AttachmentID"/>
                     *     &lt;/xs:choice>
                     *   &lt;/xs:complexType>
                     * &lt;/xs:element>
                     * </pre>
                     */
                    public static class Media implements Serializable
                    {
                        private int mediaListSelect = -1;
                        private static final int OBJECT_ID_CHOICE = 0;
                        private static final int MEDIA_LINK_CHOICE = 1;
                        private static final int ATTACHMENT_ID_CHOICE = 2;
                        private UniqueIDContextType objectID;
                        private MediaLink mediaLink;
                        private UniqueIDContextType attachmentID;

                        private void setMediaListSelect(int choice) {
                            if (mediaListSelect == -1) {
                                mediaListSelect = choice;
                            } else if (mediaListSelect != choice) {
                                throw new IllegalStateException(
                                        "Need to call clearMediaListSelect() before changing existing choice");
                            }
                        }

                        /** 
                         * Clear the choice selection.
                         */
                        public void clearMediaListSelect() {
                            mediaListSelect = -1;
                        }

                        /** 
                         * Check if ObjectID is current selection for choice.
                         * 
                         * @return <code>true</code> if selection, <code>false</code> if not
                         */
                        public boolean ifObjectID() {
                            return mediaListSelect == OBJECT_ID_CHOICE;
                        }

                        /** 
                         * Get the 'ObjectID' element value. Description-associated MEDIA OBJECT ID or Key. Example: IMG1
                         * 
                         * @return value
                         */
                        public UniqueIDContextType getObjectID() {
                            return objectID;
                        }

                        /** 
                         * Set the 'ObjectID' element value. Description-associated MEDIA OBJECT ID or Key. Example: IMG1
                         * 
                         * @param objectID
                         */
                        public void setObjectID(UniqueIDContextType objectID) {
                            setMediaListSelect(OBJECT_ID_CHOICE);
                            this.objectID = objectID;
                        }

                        /** 
                         * Check if MediaLink is current selection for choice.
                         * 
                         * @return <code>true</code> if selection, <code>false</code> if not
                         */
                        public boolean ifMediaLink() {
                            return mediaListSelect == MEDIA_LINK_CHOICE;
                        }

                        /** 
                         * Get the 'MediaLink' element value. Description-associated MEDIA LINK.
                         * 
                         * @return value
                         */
                        public MediaLink getMediaLink() {
                            return mediaLink;
                        }

                        /** 
                         * Set the 'MediaLink' element value. Description-associated MEDIA LINK.
                         * 
                         * @param mediaLink
                         */
                        public void setMediaLink(MediaLink mediaLink) {
                            setMediaListSelect(MEDIA_LINK_CHOICE);
                            this.mediaLink = mediaLink;
                        }

                        /** 
                         * Check if AttachmentID is current selection for choice.
                         * 
                         * @return <code>true</code> if selection, <code>false</code> if not
                         */
                        public boolean ifAttachmentID() {
                            return mediaListSelect == ATTACHMENT_ID_CHOICE;
                        }

                        /** 
                         * Get the 'AttachmentID' element value. Description-associated Media Attachment ID. Example: ATT1.
                         * 
                         * @return value
                         */
                        public UniqueIDContextType getAttachmentID() {
                            return attachmentID;
                        }

                        /** 
                         * Set the 'AttachmentID' element value. Description-associated Media Attachment ID. Example: ATT1.
                         * 
                         * @param attachmentID
                         */
                        public void setAttachmentID(
                                UniqueIDContextType attachmentID) {
                            setMediaListSelect(ATTACHMENT_ID_CHOICE);
                            this.attachmentID = attachmentID;
                        }
                    }
                }
                /** 
                 * Seat Description Key-Value Word Pairs.
                Examples:
                ReclineDegree - 25
                BedLengthLowEnd - 68
                BedLengthHighEnd - 78
                WidthLowEnd - 20
                WidthHighEnd - 21.5
                PitchLowEnd - 32
                PitchHighEnd - 34.4

                 * 
                 * Schema fragment(s) for this class:
                 * <pre>
                 * &lt;xs:element xmlns:ns="http://www.iata.org/IATA/EDIST/2017.2" xmlns:xs="http://www.w3.org/2001/XMLSchema" name="Keywords" minOccurs="0">
                 *   &lt;xs:complexType>
                 *     &lt;xs:sequence>
                 *       &lt;xs:element type="ns:KeyWordType" name="KeyWord" maxOccurs="unbounded"/>
                 *     &lt;/xs:sequence>
                 *   &lt;/xs:complexType>
                 * &lt;/xs:element>
                 * </pre>
                 */
                public static class Keywords implements Serializable
                {
                    private List<KeyWordType> keyWordList = new ArrayList<KeyWordType>();

                    /** 
                     * Get the list of 'KeyWord' element items. Key-Value detail.
                     * 
                     * @return list
                     */
                    public List<KeyWordType> getKeyWordList() {
                        return keyWordList;
                    }

                    /** 
                     * Set the list of 'KeyWord' element items. Key-Value detail.
                     * 
                     * @param list
                     */
                    public void setKeyWordList(List<KeyWordType> list) {
                        keyWordList = list;
                    }
                }
            }
        }
    }
}
