package com.gridnine.xtrip.server.ibecorp.tasks.groovy.xtr13639

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.Xeption
import com.gridnine.xtrip.common.model.booking.BookingFile
import com.gridnine.xtrip.common.model.booking.BookingFileIndex
import com.gridnine.xtrip.common.model.booking.StatisticalData
import com.gridnine.xtrip.common.model.booking.TravelSubject
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.handlers.ProductHandler
import com.gridnine.xtrip.common.model.profile.Organization
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.search.utils.SearchQueryHelper
import com.gridnine.xtrip.common.util.CollectionUtil
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date());
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm");

@Field boolean stopWhenError = true;

@Field List<EntityReference<Organization>> orgRefs = null;
//@Field List<EntityReference<Organization>> orgRefs = Arrays.asList(
//        new EntityReference<>("54842587-e729-4abf-b79e-b1031f65ea35", Organization.class, null),
//)

doJob()


void doJob(){
    fileMessage("Останавливать процесс при ошибке? ${stopWhenError ? 'Да' : 'Нет'}");
    fileMessage("");

    SearchQuery query = new SearchQuery()
    query.getCriteria().getCriterions().add(
            SearchCriterion.ge(BookingFileIndex.Property.createDate.name(), MiscUtil.createDate(2023, 9, 1)))

    if (CollectionUtil.isNotEmpty(orgRefs)) {
        List<String> orgNames = new ArrayList<>()
        orgRefs.each {
            EntityContainer<Organization> orgCtr = EntityStorage.get().resolve(it);
            if (orgCtr == null) {
                throw Xeption.forEndUser("Невалидная ссылка на организацию {0}", it.getUid());
            }
            orgNames.add(orgCtr.getEntity().toString())
        }
        fileMessage("Фильтр по организациям ${orgNames}");
        query.getCriteria().getCriterions().add(
                SearchCriterion.or(
                        SearchQueryHelper.buildOrEqEntityCriterion(BookingFileIndex.Property.client.name(), orgRefs),
                        SearchQueryHelper.buildOrContainsEntityCriterion(BookingFileIndex.Property.subagencies.name(), orgRefs),
                        SearchQueryHelper.buildOrEqEntityCriterion(BookingFileIndex.Property.agency.name(), orgRefs)
                ))
    }

    query.getPreferredProperties().add("containerUid");

    List<BookingFileIndex> indices = EntityStorage.get().search(BookingFileIndex.class, query).getData();

    fileMessage("Найдено ${indices.size()} заказов");

    if (indices.isEmpty()) {
        return;
    }

    def total = indices.size()
    def current = 0
    def progr = 0
    int errors = 0

    fileMessage("");

    Map<TravelSubject, UdidsData> resultMap = new HashMap<>()

    for (BookingFileIndex idx : indices){
        if (isToBeStopped()) {
            fileMessage("iterator stopped")
            break;
        }
        try {
            EntityContainer<BookingFile> ctr = EntityStorage.get().resolve(idx.source)
            if (ctr != null) {
                processBooking(ctr.getEntity(), resultMap)
            }
        } catch (Exception ex) {
            errors++;
            fileMessage("Ошибка (${idx.getSource().caption}/${idx.getSource().uid}) : ${TextUtil.getExceptionStackTrace(ex)}", false);
            if (stopWhenError){
                fileMessage("Процесс остановлен из-за ошибки в заказе ${idx.getSource().caption} (${idx.getSource().uid})");
                fileMessage(TextUtil.getExceptionStackTrace(ex))
                break
            }
        }

        def progrNew = Math.round(current * 100d / total)
        if (progr != progrNew) {
            progress("${current + 1} / ${total}".toString(), (int) progrNew)
        }
        progr = progrNew
        current++
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("");
    fileMessage("");
    fileMessage("Итого обработано ${current} заказов.");
    fileMessage("");
    fileMessage("Возникло ошибок: ${errors}");
    fileMessage("");
    resultMap.each {
        fileMessage("${it.key}: " +
                "Всего: ${it.value.total}, " +
                "Без Удидов: ${it.value.withoutUdids}, " +
                "Удиды в обеих коллекциях: ${it.value.udidsInBothCollections}, " +
                "Удиды только в неверной коллекции costCodes: ${it.value.udidsOnlyInWrongSingleCollection}, " +
                "Удиды только в неверной коллекции travellerCostCodes: ${it.value.udidsOnlyInWrongMultiCollection}");
        fileMessage("Заказы с Удидами в обеих коллекциях: ${it.value.udidsInBothCollectionsBookings}", false);
        fileMessage("Заказы с Удидами только в неверной коллекции costCodes: ${it.value.udidsOnlyInWrongSingleCollectionBookings}", false);
        fileMessage("Заказы с Удидами только в неверной коллекции travellerCostCodes: ${it.value.udidsOnlyInWrongMultiCollectionBookings}", false);
        fileMessage("", false);
    }
    fileMessage("");
    fileMessage("Итого");
    fileMessage("Удиды в обеих коллекциях: ${resultMap.values().collect {it.udidsInBothCollections}.sum()}");
    fileMessage("Удиды только в неверной коллекции costCodes: ${resultMap.values().collect{it.udidsOnlyInWrongSingleCollection}.sum()}");
    fileMessage("Удиды только в неверной коллекции travellerCostCodes: ${resultMap.values().collect{it.udidsOnlyInWrongMultiCollection}.sum()}");
}

void processBooking(BookingFile booking, Map<TravelSubject, UdidsData> resultMap) {
    booking.getReservations().collectMany {it.products}.each {pr ->
        ProductHandler ph = ProductHandler.of(pr)
        TravelSubject travelSubject = ph.findTravelSubject(pr);
        UdidsData data = resultMap.get(travelSubject)
        if (data == null) {
            data = new UdidsData()
            resultMap.put(travelSubject, data)
        }
        data.total++
        StatisticalData statisticalData = ph.getStatisticalData(pr)
        if (statisticalData == null || (statisticalData.getCostCodes().isEmpty() && statisticalData.getTravellerCostCodes().isEmpty())) {
            data.withoutUdids++
        } else if (!statisticalData.getCostCodes().isEmpty() && !statisticalData.getTravellerCostCodes().isEmpty()){
            data.udidsInBothCollections++
            data.udidsInBothCollectionsBookings += booking.number
        } else {
            if ((!ph.isMultiTraveller() && !statisticalData.getTravellerCostCodes().isEmpty())){
                data.udidsOnlyInWrongSingleCollection++
                data.udidsOnlyInWrongSingleCollectionBookings += booking.number
            }
            if ((ph.isMultiTraveller() && !statisticalData.getCostCodes().isEmpty())){
                data.udidsOnlyInWrongMultiCollection++
                data.udidsOnlyInWrongMultiCollectionBookings += booking.number
            }
        }
    }
}

class UdidsData {

    int total;
    int withoutUdids;
    int udidsInBothCollections;
    int udidsOnlyInWrongSingleCollection;
    int udidsOnlyInWrongMultiCollection;
    List<String> udidsInBothCollectionsBookings = new ArrayList<>();
    List<String> udidsOnlyInWrongSingleCollectionBookings = new ArrayList<>();
    List<String> udidsOnlyInWrongMultiCollectionBookings = new ArrayList<>();
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str);
    }
    new File(Environment.getTempFolder(), "xtr13639-${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n");
}

