package com.gridnine.xtrip.server.railway.model.helpers

import com.gridnine.xtrip.common.model.helpers.DictHelper
import com.gridnine.xtrip.common.model.helpers.SystemHelper
import com.gridnine.xtrip.common.railway.model.helpers.RailwayProductHelper
import com.gridnine.xtrip.common.util.EnumUtil
import groovy.transform.Field
import com.gridnine.xtrip.common.model.booking.ProductStatus
import com.gridnine.xtrip.common.model.booking.TransportationType
import com.gridnine.xtrip.common.model.booking.VatBasisType
import com.gridnine.xtrip.common.model.booking.VatComponent
import com.gridnine.xtrip.common.model.booking.VatDetalization
import com.gridnine.xtrip.common.model.booking.railway.RailwayProduct
import com.gridnine.xtrip.common.model.booking.railway.RailwaySegment
import com.gridnine.xtrip.common.model.dict.Country
import com.gridnine.xtrip.common.model.dict.DictionaryReference
import com.gridnine.xtrip.common.model.dict.railway.RailwayStation
import com.gridnine.xtrip.common.railway.model.helpers.RailwayProductDictionaryHelper
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil

import java.math.RoundingMode
import java.time.LocalDate
import java.time.format.DateTimeFormatter
import java.util.function.BinaryOperator


@Field final  static Map<VatBasisType, List<MiscUtil.Pair<Date, Double>>> rusVatRates =
        getRusVatRatesMap()

@Field final  static Map<VatBasisType, List<MiscUtil.Pair<Date, Double>>> kzVatRates =
        getKzVatRatesMap()

static def getKzVatRatesMap() {
    Map<VatBasisType, List<MiscUtil.Pair<Date, Double>>> result =
            new HashMap()
    for (VatBasisType type : VatBasisType.values()) {
        result.put(type, new ArrayList<>())
    }
}

static def getRusVatRatesMap(){
    DateTimeFormatter format = DateTimeFormatter.ofPattern("yyyy-MM-dd") //$NON-NLS-1$
     Map<VatBasisType, List<MiscUtil.Pair<Date, Double>>> result =
            new HashMap<VatBasisType, List<MiscUtil.Pair<Date, Double>>>()
    for (VatBasisType type : VatBasisType.values()) {
        result.put(type, new ArrayList<>())
    }

    result.get(VatBasisType.FARE)
            .add(new MiscUtil.Pair<Date, Double>(
                    MiscUtil.toDate(LocalDate.parse("2000-01-01", format)), //$NON-NLS-1$
                    18d))
    result.get(VatBasisType.FARE)
            .add(new MiscUtil.Pair<Date, Double>(
                    MiscUtil.toDate(LocalDate.parse("2016-01-01", format)), //$NON-NLS-1$
                    10d))
    result.get(VatBasisType.FARE)
            .add(new MiscUtil.Pair<Date, Double>(
                    MiscUtil.toDate(LocalDate.parse("2017-01-01", format)), //$NON-NLS-1$
                    0d))

    result.get(VatBasisType.SERVICE)
            .add(new MiscUtil.Pair<Date, Double>(
                    MiscUtil.toDate(LocalDate.parse("2000-01-01", format)), //$NON-NLS-1$
                    18d))
    result.get(VatBasisType.SERVICE)
            .add(new MiscUtil.Pair<Date, Double>(
                    MiscUtil.toDate(LocalDate.parse("2019-01-01", format)), //$NON-NLS-1$
                    20d))
    return result
}

static Map<VatBasisType, List<MiscUtil.Pair<Date, Double>>> getVatRates() {
    if (SystemHelper.isKazakhInstallation()) {
        return kzVatRates
    }
    return rusVatRates
}

static def getVatRate(final VatBasisType type, final Date date) {

    return getVatRates()[type].stream()
            .reduce(new MiscUtil.Pair<Date, Double>(null, MiscUtil.guarded(DictHelper.getDefaultVat(date)).doubleValue()),
                    new BinaryOperator<MiscUtil.Pair<Date, Double>>() {
                        @Override
                        MiscUtil.Pair<Date, Double> apply(MiscUtil.Pair<Date, Double> reduce, MiscUtil.Pair<Date, Double> item) {
                           return  (MiscUtil.compare(item.getFirst(), reduce.getFirst()) >= 0)&& (MiscUtil.compare(item.getFirst(), date) <= 0)? item: reduce
                        }
                    }).getSecond()
}

static def getTransportationType(List<RailwaySegment> segments) {

    boolean domestic = false
    boolean international = false

    for (RailwaySegment segment : segments) {

        DictionaryReference<RailwayStation> departureLocation =
                segment.getDepartureStation()
        DictionaryReference<RailwayStation> arriveLocation =
                segment.getArriveStation()

        Country departureCountry =
                RailwayProductDictionaryHelper.getCountry(departureLocation)
        Country arriveCountry =
                RailwayProductDictionaryHelper.getCountry(arriveLocation)

        if ((arriveCountry == null) || (departureCountry == null)) {
            continue
        }

        if (departureCountry.isDomestic() && arriveCountry.isDomestic()) {

            domestic = true
            continue
        }

        international = true
    }

    if (domestic && international) {
        return TransportationType.COMBINED
    } else if (domestic) {
        return TransportationType.DOMESTIC
    } else if (international) {
        return TransportationType.INTERNATIONAL
    } else {
        return TransportationType.NONE
    }
}

static def getTransportationType(final RailwayProduct product) {

    if (product == null) {
        return null
    }

    return getTransportationType(product.getSegments())
}

static def isSuburbanTrainNumber(String trainNumber){
    if(TextUtil.isBlank(trainNumber)){
        return false
    }
    try{
        int tn = Integer.parseInt(trainNumber)
        return tn > 7000 && tn< 8000
    } catch (Exception ignored){
        return false
    }
}

static def isSuburbanTrain(RailwayProduct product){ 
    return product.getSegments().any {isSuburbanTrainNumber(it.getTrainNumber())}
}

static def isHasVat(RailwayProduct product){

    if (product == null || isSuburbanTrain(product)) {
        return false
    }


    RailwayProduct sellProduct = product

    if ((product.getStatus() == ProductStatus.REFUND)) {

        sellProduct = product.getPreviousProduct()

        if (sellProduct == null) {
            sellProduct = product
        }
    }

    return getTransportationType(sellProduct) == TransportationType.DOMESTIC
}

static def calculateVat(final BigDecimal total,final double rate) {
    return (total * BigDecimal.valueOf(rate)).divide(
            BigDecimal.valueOf(100).add(BigDecimal.valueOf(rate)), 2,
            RoundingMode.HALF_UP)
}

static def isHasPenaltyVat(final RailwayProduct product) {
    return (!SystemHelper.isKazakhInstallation() && !isSuburbanTrain(product)) || isHasVat(product);
}
static def updateVatDetalization(RailwayProduct product, VatDetalization detalization){

    BigDecimal tariff = MiscUtil.guarded(product.getEquivalentFare() as BigDecimal)
    BigDecimal service = MiscUtil.guarded(product.getServiceFare() as BigDecimal)
    BigDecimal penalty = MiscUtil.guarded(product.getPenalty() as BigDecimal)

    Date issueDate = product.getIssueDate()
    if(product.getStatus() == ProductStatus.REFUND){
        issueDate = product.getPreviousProduct().getIssueDate()
    }
    boolean hasVat = isHasVat(product)
    if (BigDecimal.ZERO != tariff) {
        VatComponent comp = new VatComponent()
        comp.setBasis(tariff)
        comp.getBasisTypes().add(VatBasisType.FARE)
        double rate = !hasVat || issueDate == null ? 0 :
                getVatRate(
                        VatBasisType.FARE, issueDate)
        comp.setRate(rate)
        comp.setSum(calculateVat(tariff, rate))
        detalization.getComponents().add(comp)
    }
    if (BigDecimal.ZERO != service) {
        VatComponent comp = new VatComponent()
        comp.setBasis(service)
        comp.getBasisTypes().add(VatBasisType.SERVICE)
        double rate = hasVat ? getVatRate(
                VatBasisType.SERVICE, issueDate) : 0
        comp.setRate(Double.valueOf(rate))
        comp.setSum(calculateVat(service, rate))
        detalization.getComponents().add(comp)
    }

    if (BigDecimal.ZERO != penalty) {
        VatComponent comp = new VatComponent()
        comp.setBasis(penalty)
        boolean refund = (product.getStatus() == ProductStatus.REFUND) || (product.getStatus() == ProductStatus.EXCHANGE)
        if (refund) {
            comp.setBasis(comp.getBasis().negate())
        }
        comp.getBasisTypes().add(VatBasisType.PENALTY)
        double rate = isHasPenaltyVat(product) ? getVatRate(VatBasisType.SERVICE, issueDate):0
        comp.setRate(Double.valueOf(rate))
        comp.setSum(calculateVat(penalty, rate))
        if (refund) {
            comp.setSum(comp.getSum().negate())
        }
        detalization.getComponents().add(comp)
    }

    def deductionVatBasisType = EnumUtil.fromNameSafe(VatBasisType.class, "DEDUCTION", null)
    if (deductionVatBasisType != null) {
        BigDecimal deduction = MiscUtil.guarded(RailwayProductHelper.getDeduction(product) as BigDecimal)
        BigDecimal deductionVat = RailwayProductHelper.getDeductionVat(product) as BigDecimal
        if (BigDecimal.ZERO != deduction) {
            VatComponent comp = new VatComponent()
            comp.setBasis(deduction.negate())
            boolean refund = (product.getStatus() == ProductStatus.REFUND) || (product.getStatus() == ProductStatus.EXCHANGE)
            if (refund) {
                comp.setBasis(comp.getBasis().negate())
            }
            comp.getBasisTypes().add(deductionVatBasisType)
            if (deductionVat != null) {
                comp.setSum(deductionVat)
                comp.setRate(deductionVat.multiply(BigDecimal.valueOf(100))
                        .divide(deduction.subtract(deductionVat), 0, BigDecimal.ROUND_HALF_UP)
                        .doubleValue())
                if (refund) {
                    comp.setSum(comp.getSum().negate())
                }
            }
            detalization.getComponents().add(comp)
        }
    }
}

updateVatDetalization(productProperty, detalizationProperty)

