//file:noinspection GrDeprecatedAPIUsage
//file:noinspection GroovyAssignabilityCheck
//file:noinspection DuplicatedCode
package com.gridnine.xtrip.server.support.model.stats.scripts

import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.asset.AssetsStorage
import com.gridnine.xtrip.common.model.booking.commission.ProductType
import com.gridnine.xtrip.common.model.booking.statistics.ProductBillingDataChangeDataSource
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.search.Projection
import com.gridnine.xtrip.common.search.ProjectionQuery
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.support.model.licence2.MomLicenceCard
import com.gridnine.xtrip.common.support.model.stats.*
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import com.gridnine.xtrip.server.db.storage.LogicalStorage
import org.apache.commons.lang.time.DateUtils

import java.text.DateFormat
import java.text.SimpleDateFormat
import java.time.temporal.TemporalAdjusters

createStyle(name: 'title', fontBold: true, h_span: 6, h_alignment: 'CENTER', v_alignment: 'CENTER', fontHeight: 14, wrapText: true)
createStyle(name: 'metadataTitle', fontBold: true, h_span: 2, h_alignment: 'RIGHT', v_alignment: 'CENTER', fontHeight: 10)
createStyle(name: 'metadataValue', parent: 'metadataTitle', h_span: 10, fontBold: false, h_alignment: 'LEFT')
createStyle(name: 'metadataDateValue', parent: 'metadataValue', format: 'm/d/yy')
createStyle(name: 'header', fontBold: false, h_alignment: 'CENTER', v_alignment: 'CENTER', fontHeight: 10,
        leftBorder: 'MEDIUM', rightBorder: 'MEDIUM', topBorder: 'MEDIUM', bottomBorder: 'MEDIUM', wrapText: true)
createStyle(name: 'data', fontBold: false, h_alignment: 'CENTER', v_alignment: 'CENTER', fontHeight: 10,
        leftBorder: 'THIN', rightBorder: 'THIN', topBorder: 'THIN', bottomBorder: 'THIN')
createStyle(name: 'textData', parent: 'data')
createStyle(name: 'fakeText', fontHeight: 10)
createStyle(name: 'numberData', parent: 'data', h_alignment: 'RIGHT')
createStyle(name: 'dateData', parent: 'data', h_alignment: 'RIGHT', format: 'm/d/yy')
createStyle(name: 'statusGroupTitle', parent: 'header', h_span: 13, h_alignment: 'RIGHT', fontBold: true)
createStyle(name: 'statusGroupNumber', parent: 'statusGroupTitle', h_span: 1)
createStyle(name: 'subagencyGroupTitle', parent: 'statusGroupTitle')
createStyle(name: 'subagencyGroupNumber', parent: 'subagencyGroupTitle', h_span: 1)
createStyle(name: 'totalTitle', parent: 'subagencyGroupTitle')
createStyle(name: 'totalNumber', parent: 'totalTitle', h_span: 1)

class ReportEntry {
    String period
    String license
    String agency
    String carrierCode
    String productType
    String status
    String gds
    String isOnline
    int count
}

class ReportPeriod {
    String caption
    Date startDate
    Date endDate
}

def periods = []
def hasAgency = (Boolean.TRUE == parameters.groupByAgency)
def splitByMonth = (Boolean.TRUE == parameters.splitByMonth)
def splitByCarrier = (Boolean.TRUE == parameters.splitByCarrier)
def splitByOnlineOffline = (Boolean.TRUE == parameters.splitByOnlineOffline)

parameters.startDate = MiscUtil.clearTime(parameters.startDate)
parameters.endDate = MiscUtil.setDayEndTime(parameters.endDate)

def dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")
def lastArchiveDate = null
try {
    String dateStr = LogicalStorage.get().getDbPropertiesStorage().getValue("product-statistics-archive")
    lastArchiveDate = dateFormat.parse(dateStr)
} catch (Exception ignored) {}

if (lastArchiveDate != null) {
    def cleanStart = DateUtils.truncate(parameters.startDate, Calendar.MONTH)
    def cleanEnd = MiscUtil.toDate(MiscUtil.toLocalDateTime(
            parameters.endDate).with(TemporalAdjusters.lastDayOfMonth()))

    if (parameters.startDate.before(lastArchiveDate) && !Objects.equals(parameters.startDate, cleanStart)) {
        parameters.startDate = cleanStart
        warn(String.format(
                "Дата начала находится внутри архивного периода и будет округлена до ближайшего начала месяца: %s",
                dateFormat.format(cleanStart)))
    }
    if (parameters.endDate.before(lastArchiveDate) && !Objects.equals(parameters.endDate, cleanEnd)) {
        parameters.endDate = cleanEnd
        warn(String.format(
                "Дата окончания находится внутри архивного периода и будет округлена до ближайшего окончания месяца: %s",
                dateFormat.format(cleanEnd)))
    }
}

if (splitByMonth) {
    Calendar cal = Calendar.getInstance()
    cal.setTime(parameters.startDate)
    Date endDate = parameters.endDate ? MiscUtil.setDayEndTime(parameters.endDate) : MiscUtil.setDayEndTime(new Date())
    while (cal.getTime().before(endDate)) {
        Date periodStartDate = MiscUtil.clearTime(cal.getTime())
        cal.add(Calendar.MONTH, 1)
        cal.set(Calendar.DAY_OF_MONTH, 1)
        cal.add(Calendar.DAY_OF_MONTH, -1)
        Date periodEndDate = MiscUtil.setDayEndTime(cal.getTime())
        if (periodEndDate.after(endDate)) {
            periodEndDate = endDate
        }
        String caption = ""
        switch (cal.get(Calendar.MONTH)) {
            case Calendar.JANUARY:
                caption = "01_Jan"
                break
            case Calendar.FEBRUARY:
                caption = "02_Feb"
                break
            case Calendar.MARCH:
                caption = "03_Mar"
                break
            case Calendar.APRIL:
                caption = "04_Apr"
                break
            case Calendar.MAY:
                caption = "05_May"
                break
            case Calendar.JUNE:
                caption = "06_Jun"
                break
            case Calendar.JULY:
                caption = "07_Jul"
                break
            case Calendar.AUGUST:
                caption = "08_Aug"
                break
            case Calendar.SEPTEMBER:
                caption = "09_Sep"
                break
            case Calendar.OCTOBER:
                caption = "10_Oct"
                break
            case Calendar.NOVEMBER:
                caption = "11_Nov"
                break
            case Calendar.DECEMBER:
                caption = "12_Dec"
                break
        }
        caption = "${cal.get(Calendar.YEAR)}_${caption}"
        cal.add(Calendar.DAY_OF_MONTH, 1)
        periods.add(new ReportPeriod(startDate: periodStartDate, endDate: periodEndDate, caption: caption))
    }
} else {
    periods.add(new ReportPeriod(startDate: MiscUtil.clearTime(parameters.startDate),
            endDate: MiscUtil.setDayEndTime(parameters.endDate), caption: "all_period"))
}

def agenciesSet = new HashSet<String>()
def carrierSet = new HashSet<String>()
def entries = new ArrayList<ReportEntry>()

periods.forEach { ReportPeriod period ->
    ProjectionQuery pq = new ProjectionQuery()
    if (period.startDate != null) {
        pq.getCriteria().getCriterions().add(SearchCriterion.ge(ProductStatistics.Property.issueDate.name(),
                MiscUtil.clearTime(period.startDate)))
    }
    if (period.endDate != null) {
        pq.getCriteria().getCriterions().add(SearchCriterion.le(ProductStatistics.Property.issueDate.name(),
                MiscUtil.setDayEndTime(period.endDate)))
    }

    ProductStatisticsActionSourceRef sourceRef = AssetsStorage.get().load(
            ProductStatisticsActionSourceRef.class, ProductBillingDataChangeDataSource.CLIENT.name())
    ProductStatisticsProductTypeRef additionalServiceTypeRef = AssetsStorage.get().load(
            ProductStatisticsProductTypeRef.class, ProductType.ADDITIONAL_SERVICE.name())
    if (sourceRef != null) {
        pq.getCriteria().getCriterions().add(SearchCriterion.or(
                SearchCriterion.eq(ProductStatistics.Property.actionSourceId.name(), null),
                SearchCriterion.or(
                        SearchCriterion.ne(ProductStatistics.Property.actionSourceId.name(), sourceRef.getId()),
                        SearchCriterion.and(
                                SearchCriterion.eq(ProductStatistics.Property.productTypeId.name(), additionalServiceTypeRef.getId()),
                                SearchCriterion.eq(ProductStatistics.Property.hasUsedCoupons.name(), Boolean.TRUE)
                        ))
        ))
    }

    ProductStatisticsStatusRef statRef = AssetsStorage.get().load(ProductStatisticsStatusRef.class, "INTENTION")
    ProductStatisticsStatusRef voidIntentionRef = AssetsStorage.get().load(ProductStatisticsStatusRef.class, "VOID_INTENTION")
    if (statRef != null) {
        if (voidIntentionRef != null) {
            pq.getCriteria().getCriterions().add(SearchCriterion.or(
                    SearchCriterion.eq(ProductStatistics.Property.statusId.name(), null),
                    SearchCriterion.and(
                            SearchCriterion.ne(ProductStatistics.Property.statusId.name(), statRef.getId()),
                            SearchCriterion.ne(ProductStatistics.Property.statusId.name(), voidIntentionRef.getId())
                    )
            ))
        } else {
            pq.getCriteria().getCriterions().add(SearchCriterion.or(
                    SearchCriterion.eq(ProductStatistics.Property.statusId.name(), null),
                    SearchCriterion.ne(ProductStatistics.Property.statusId.name(), statRef.getId())))
        }
    }

    if (parameters.license != null && !parameters.license.isEmpty()) {
        List<Short> ids = new ArrayList<>()
        parameters.license.forEach { EntityReference<MomLicenceCard> licenseRef ->
            ProductStatisticsLicenseRef license = AssetsStorage.get().load(ProductStatisticsLicenseRef.class, licenseRef.getUid())
            if (license != null) {
                ids.add(license.id)
            } else if (ids.isEmpty()) {
                ids.add(-1 as Short)
            }
        }
        pq.getCriteria().getCriterions().add(SearchCriterion.in(
                ProductStatistics.Property.licenseId.name(), ids.toArray(new short[ids.size()])))
    }

    pq.getCriteria().getCriterions().add(SearchCriterion.eq(
            ProductStatistics.Property.modifiedAfterFreezeDate.name(), Boolean.FALSE))

    pq.projections.add(Projection.group(ProductStatistics.Property.licenseId.name(), "_licenseId"))
    pq.projections.add(Projection.count(ProductStatistics.Property.productTypeId.name(), "_count"))
    pq.projections.add(Projection.group(ProductStatistics.Property.productTypeId.name(), "_productTypeId"))
    pq.projections.add(Projection.group(ProductStatistics.Property.gdsId.name(), "_gdsId"))
    pq.projections.add(Projection.group(ProductStatistics.Property.statusId.name(), "_statusId"))
    pq.projections.add(Projection.group(ProductStatistics.Property.agencyId.name(), "_agencyId"))
    pq.projections.add(Projection.group(ProductStatistics.Property.carrierCode.name(), "_carrierCode"))
    pq.projections.add(Projection.group(ProductStatistics.Property.actionSourceId.name(), "_actionSourceId"))

    SearchQuery sq = new SearchQuery()
    sq.getCriteria().getCriterions().addAll(pq.getCriteria().getCriterions())

    def data = AssetsStorage.get().search(ProductStatistics.class, pq).getData()
    def archiveData = AssetsStorage.get().search(ProductStatisticsArchive.class, sq).getData()
    mergeArchiveData(data, archiveData)

    data.each { Map<String, Object> entry ->
        def re = new ReportEntry()
        short licence = entry.get("_licenseId") as short
        int count = entry.get("_count") as int
        byte type = entry.get("_productTypeId") as byte
        byte gds = entry.get("_gdsId") as byte
        byte status = entry.get("_statusId") as byte

        Integer agency = entry.get("_agencyId")
        if (agency != null) {
            def agencyRef = AssetsStorage.get().find(
                    ProductStatisticsAgencyRef.class, ProductStatisticsAgencyRef.Property.id.name(), agency)
            re.agency = agencyRef?.caption
        }
        if (!re.agency) {
            re.agency = "Не задано"
        }

        re.carrierCode = entry.get("_carrierCode")
        if (!re.carrierCode) {
            re.carrierCode = "Не задано"
        }

        def gdsRef = AssetsStorage.get().find(
                ProductStatisticsGdsRef.class, ProductStatisticsGdsRef.Property.id.name(), gds)
        re.gds = gdsRef?.caption
        if (gdsRef != null && "UNKNOWN" == gdsRef.uid) {
            return
        }

        def typeRef = AssetsStorage.get().find(
                ProductStatisticsProductTypeRef.class, ProductStatisticsProductTypeRef.Property.id.name(), type)
        re.productType = typeRef?.caption
        if (typeRef != null && "UNKNOWN" == typeRef.uid) {
            return
        }
        if (typeRef != null && "INSURANCE_ACCIDENT" == typeRef.uid && gdsRef != null && "GABRIEL" == gdsRef.uid) {
            return
        }
        if (typeRef != null && "MISC_PRODUCT" == typeRef.uid) {
            return
        }
        //noinspection SpellCheckingInspection
        if (typeRef != null && "EXCESS_BAGAGE" == typeRef.uid) {
            return
        }
        if (typeRef != null && "PRODUCT_VOIDING" == typeRef.uid) {
            return
        }

        def statusRef = AssetsStorage.get().find(
                ProductStatisticsStatusRef.class, ProductStatisticsStatusRef.Property.id.name(), status)
        re.status = statusRef?.caption

        def licenseRef = AssetsStorage.get().find(
                ProductStatisticsLicenseRef.class, ProductStatisticsLicenseRef.Property.id.name(), licence)
        def licenseCaption = licenseRef?.caption
        if (licenseRef != null) {
            def licenceCtr = EntityStorage.get().load(MomLicenceCard.class, licenseRef.uid)
            if (licenceCtr != null && licenceCtr.entity.customerName != null) {
                licenseCaption = licenceCtr.entity.customerName
            }
        }
        re.license = licenseCaption

        Byte actionSource = entry.get("_actionSourceId")
        if (actionSource != null) {
            def actionSourceRef = AssetsStorage.get().find(
                    ProductStatisticsActionSourceRef.class, ProductStatisticsActionSourceRef.Property.id.name(), actionSource)
            if (actionSourceRef != null) {
                re.isOnline = ("ONLINE" == actionSourceRef.uid) ? "Онлайн" : "Офлайн"
            }
        }

        re.count = count
        re.period = period.caption
        entries.add(re)

        agenciesSet.add(re.agency)
        carrierSet.add(re.carrierCode)
    }
}

Collections.sort(entries, { ReportEntry e1, ReportEntry e2 ->
    int res = MiscUtil.compare(e1.period, e2.period)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.license, e2.license)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.agency, e2.agency)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.carrierCode, e2.carrierCode)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.productType, e2.productType)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.gds, e2.gds)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.status, e2.status)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.isOnline, e2.isOnline)
    if (res != 0) {
        return res
    }
    return 0
})

page { "Данные" } {

    columnWidth(20); text('Лицензия', 'header'); nextColumn()
    if (splitByMonth) {
        columnWidth(20); text('Период', 'header'); nextColumn()
    }
    if (hasAgency) {
        columnWidth(20); text('Агентство', 'header'); nextColumn()
    }
    if (splitByCarrier) {
        columnWidth(20); text('Авиакомпания', 'header'); nextColumn()
    }
    columnWidth(20); text('Канал продаж', 'header'); nextColumn()
    if (splitByOnlineOffline) {
        columnWidth(20); text('Онлайн/Офлайн', 'header'); nextColumn()
    }
    columnWidth(20); text('Тип продукта', 'header'); nextColumn()
    columnWidth(20); text('Статус', 'header'); nextColumn()
    columnWidth(20); text('Кол-во', 'header')

    entries.forEach { entry ->
        nextRow()
        text(entry.license, 'textData'); nextColumn()
        if (splitByMonth) {
            text(entry.period, 'textData'); nextColumn()
        }
        if (hasAgency) {
            text(entry.agency, 'textData'); nextColumn()
        }
        if (splitByCarrier) {
            text(entry.carrierCode, 'textData'); nextColumn()
        }
        text(entry.gds, 'textData'); nextColumn()
        if (splitByOnlineOffline) {
            text(entry.isOnline, 'textData'); nextColumn()
        }
        text(entry.productType, 'textData'); nextColumn()
        text(entry.status, 'textData'); nextColumn()
        number(entry.count, 'numberData')
    }

}

static def mergeArchiveData(List<Map<String, Object>> data, List<ProductStatisticsArchive> archiveData) {
    outer:
    for (ProductStatisticsArchive archive : archiveData) {
        for (Map<String, Object> entry : data) {
            if (isMergeCandidate(entry, archive)) {
                int count = entry.get("_count") as int
                entry.put("_count", count + archive.getCount())
                continue outer
            }
        }
        data.add(toMap(archive))
    }
}

static def isMergeCandidate(Map<String, Object> entry, ProductStatisticsArchive archive) {
    // NOTNULL
    if (entry.get("_licenseId") as short != archive.licenseId) {
        return false
    }
    if (entry.get("_productTypeId") as byte != archive.productTypeId) {
        return false
    }
    if (entry.get("_gdsId") as byte != archive.gdsId) {
        return false
    }
    if (entry.get("_statusId") as byte != archive.statusId) {
        return false
    }

    // NULLABLE
    Byte actionSourceId = entry.get("_actionSourceId")
    if (actionSourceId != archive.actionSourceId) {
        return false
    }
    Integer agencyId = entry.get("_agencyId")
    if (agencyId != archive.agencyId) {
        return false
    }
    String carrierCode = entry.get("_carrierCode")
    if (carrierCode != archive.carrierCode) {
        return false
    }

    return true
}

static def toMap(ProductStatisticsArchive archive) {
    Map<String, Object> entry = new HashMap<>()

    entry.put("_licenseId", archive.getLicenseId())
    entry.put("_productTypeId", archive.getProductTypeId())
    entry.put("_gdsId", archive.getGdsId())
    entry.put("_statusId", archive.getStatusId())
    entry.put("_agencyId", archive.getAgencyId())
    entry.put("_carrierCode", archive.getCarrierCode())
    entry.put("_actionSourceId", archive.getActionSourceId())
    entry.put("_count", archive.getCount())

    return entry
}

def findWidth = { caption, clb ->
    def width = caption.length()
    entries.forEach { et ->
        def content = clb.call(et)
        if (content != null) {
            if (width < content.length()) {
                width = content.length()
            }
        }
    }
    return (int) width * 0.9 + 2
}

def getReportPeriod = { Date periodBegin, Date periodEnd ->
    String reportPeriod = ""
    DateFormat df = new SimpleDateFormat("dd",
            new Locale("ru", "RU"))
    DateFormat df2 = new SimpleDateFormat("dd.MM.yy",
            new Locale("ru", "RU"))
    if ((periodBegin != null) && (periodEnd != null)) {
        if ((periodBegin.getDate() == periodEnd.getDate())
                && (periodBegin.getMonth() == periodEnd.getMonth())
                && (periodBegin.getYear() == periodEnd.getYear())) {
            reportPeriod = df2.format(periodEnd)
        } else if ((periodBegin.getMonth() == periodEnd.getMonth())
                && (periodBegin.getYear() == periodEnd.getYear())) {
            reportPeriod = df.format(periodBegin) + "-" + df2.format(periodEnd)
        } else {
            reportPeriod = df2.format(periodBegin) + "-" + df2.format(periodEnd)

        }
    }
    return reportPeriod
}

page { "Отчет" } {

    def agencies = ""
    if (parameters.license != null && !parameters.license.isEmpty()) {
        if (parameters.license.size() < 3) {
            agencies = TextUtil.join(",", parameters.license)
        }
    }

    def period = getReportPeriod(parameters.startDate, parameters.endDate)
    def title = "Статистика продаж ${agencies} за период ${period}"

    rowAutoHeight()
    text(title, 'title', 0, 0)

    nextRow()
    rowHeight(12)

    nextRow()
    pivotTable {
        source(sheet: "Данные", leftTop: [row: 0, column: 0],
                rightBottom: [row   : entries.size(),
                              column: 4 + (hasAgency ? 1 : 0) + (splitByCarrier ? 1 : 0) + (splitByMonth ? 1 : 0) + (splitByOnlineOffline ? 1 : 0)])
        if (splitByOnlineOffline) {
            if (hasAgency) {
                if (splitByMonth) {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5, 6, 7)
                        sumColumn(8)
                    } else {
                        row(0, 1, 2, 3, 4, 5, 6)
                        sumColumn(7)
                    }
                } else {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5, 6)
                        sumColumn(7)
                    } else {
                        row(0, 1, 2, 3, 4, 5)
                        sumColumn(6)
                    }
                }
            } else {
                if (splitByMonth) {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5, 6)
                        sumColumn(7)
                    } else {
                        row(0, 1, 2, 3, 4, 5)
                        sumColumn(6)
                    }
                } else {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5)
                        sumColumn(6)
                    } else {
                        row(0, 1, 2, 3, 4)
                        sumColumn(5)
                    }
                }
            }
        } else {
            if (hasAgency) {
                if (splitByMonth) {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5, 6)
                        sumColumn(7)
                    } else {
                        row(0, 1, 2, 3, 4, 5)
                        sumColumn(6)
                    }
                } else {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5)
                        sumColumn(6)
                    } else {
                        row(0, 1, 2, 3, 4)
                        sumColumn(5)
                    }
                }
            } else {
                if (splitByMonth) {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4, 5)
                        sumColumn(6)
                    } else {
                        row(0, 1, 2, 3, 4)
                        sumColumn(5)
                    }
                } else {
                    if (splitByCarrier) {
                        row(0, 1, 2, 3, 4)
                        sumColumn(5)
                    } else {
                        row(0, 1, 2, 3)
                        sumColumn(4)
                    }
                }
            }
        }
    }

    nextRow()
    if (splitByMonth) {
        text("", "fakeText"); columnWidth(findWidth("Период", { return it.period })); nextColumn()
    }
    if (hasAgency) {
        text("", "fakeText"); columnWidth(findWidth("Агентство", { return it.agency })); nextColumn()
    }
    if (splitByCarrier) {
        text("", "fakeText"); columnWidth(findWidth("Авиакомпания", { return it.carrierCode })); nextColumn()
    }
    text("", "fakeText"); columnWidth(findWidth("Лицензия", { return it.license })); nextColumn()
    text("", "fakeText"); columnWidth(findWidth("Канал продаж", { return it.gds })); nextColumn()
    if (splitByOnlineOffline) {
        text("", "fakeText"); columnWidth(findWidth("Онлайн/Офлайн", { return it.isOnline })); nextColumn()
    }
    text("", "fakeText"); columnWidth(findWidth("Тип продукта", { return it.productType })); nextColumn()
    text("", "fakeText"); columnWidth(findWidth("Статус", { return it.status }))
}

setDefaultSheet("Отчет")
