package com.gridnine.xtrip.server.ibecorp.tasks.groovy

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.booking.BookingFile
import com.gridnine.xtrip.common.model.booking.BookingFileIndex
import com.gridnine.xtrip.common.model.booking.Reservation
import com.gridnine.xtrip.common.model.booking.ReservationType
import com.gridnine.xtrip.common.model.booking.TravelSubject
import com.gridnine.xtrip.common.model.booking.xtriphotels.HotelProduct
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.helpers.BookingHelper
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.util.TextUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date())
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm")

@Field boolean onlyLook = true

if (onlyLook) {
    fileMessage("Режим 'Только просмотр' ")
    fileMessage("")
}
doJob()


void doJob() {
    SearchQuery query = new SearchQuery()
    query.getCriteria().getCriterions().add(
            SearchCriterion.contains(BookingFileIndex.Property.travelSubject.name(), TravelSubject.HOTEL_RESERVATION))
    query.getPreferredProperties().add(BookingFileIndex.Property.fullNumber.name())

    List<BookingFileIndex> indexes = EntityStorage.get().search(BookingFileIndex.class, query).getData()

    def current = 0
    def total = indexes.size()
    def progr = 0

    fileMessage("Найдено $total отельных заказов")

    int errors = 0
    List<String> updatedBookings = new ArrayList<>()

    for (BookingFileIndex idx : indexes) {
        if (isToBeStopped()) {
            fileMessage("iterator stopped")
            break
        }
        try {
            EntityContainer<BookingFile> bookingCtr = EntityStorage.get().resolve(idx.getSource())
            if (needUpdate(bookingCtr.getEntity())) {
                if (!onlyLook) {
                    update(bookingCtr.getEntity())
                    bookingCtr.getVersionInfo().setDataSource("ibecorp-2787")
                    bookingCtr.getVersionInfo().setVersionNotes("Обнуление Online-типа бронирования")
                    EntityStorage.get().save(bookingCtr, true)
                }
                updatedBookings.add(idx.fullNumber)
            }
        } catch (Exception ex) {
            errors++
            fileMessage("Ошибка (${idx.fullNumber}) : ${TextUtil.getExceptionStackTrace(ex)}", false)
        }

        def progrNew = Math.round(current * 100d / total)
        if (progr != progrNew) {
            progress("${current + 1} / ${total}".toString(), (int) progrNew)
        }
        progr = progrNew
        current++
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("")
    fileMessage("")
    fileMessage("Итого обработано ${current} индексов.")
    fileMessage("")
    fileMessage("Изменено ${updatedBookings.size()} заказов")
    fileMessage(updatedBookings.toString(), false)
    fileMessage("")
    fileMessage("Возникло ${errors} ошибок")
}

static boolean needUpdate(BookingFile booking) {
    return updateOrCheck(booking, true)
}

static void update(BookingFile booking) {
    updateOrCheck(booking, false)
}

static boolean updateOrCheck(BookingFile booking, boolean check) {
    for (Reservation res : BookingHelper.getReservationsForProduct(booking, HotelProduct.class)) {
        if (res.getType() == ReservationType.INTERNET || res.getType() == ReservationType.IBE) {
            if (check) {
                return true
            } else {
                res.setType(null)
            }
        }
    }
    return false
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str)
    }
    new File(Environment.getTempFolder(), "ibecorp2787_${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n")
}