//file:noinspection GrDeprecatedAPIUsage
//file:noinspection GroovyAssignabilityCheck
//file:noinspection DuplicatedCode

// com/gridnine/xtrip/server/support/model/stats/scripts/redapp-counter-report.groovy

package com.gridnine.xtrip.server.support.model.stats.scripts

import com.gridnine.xtrip.common.model.BaseAsset
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.Xeption
import com.gridnine.xtrip.common.model.asset.AssetsStorage
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.search.Projection
import com.gridnine.xtrip.common.search.ProjectionQuery
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.support.model.licence2.MomLicenceCard
import com.gridnine.xtrip.common.support.model.stats.RedAppCounter
import com.gridnine.xtrip.common.support.model.stats.RedAppCounterArchive
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field FastDateFormat DF = FastDateFormat.getInstance("dd.MM.yyyy")

createStyle(name: 'title', fontBold: true, h_span: 2, h_alignment: 'CENTER', v_alignment: 'CENTER', fontHeight: 14, wrapText: true)
createStyle(name: 'metadataTitle', fontBold: true, h_span: 2, h_alignment: 'RIGHT', v_alignment: 'CENTER', fontHeight: 10)
createStyle(name: 'metadataValue', parent: 'metadataTitle', h_span: 10, fontBold: false, h_alignment: 'LEFT')
createStyle(name: 'metadataDateValue', parent: 'metadataValue', format: 'm/d/yy')
createStyle(name: 'header', fontBold: false, h_alignment: 'CENTER', v_alignment: 'CENTER', fontHeight: 10,
        leftBorder: 'MEDIUM', rightBorder: 'MEDIUM', topBorder: 'MEDIUM', bottomBorder: 'MEDIUM', wrapText: true)
createStyle(name: 'data', fontBold: false, h_alignment: 'CENTER', v_alignment: 'CENTER', fontHeight: 10,
        leftBorder: 'THIN', rightBorder: 'THIN', topBorder: 'THIN', bottomBorder: 'THIN')
createStyle(name: 'textData', parent: 'data')
createStyle(name: 'fakeText', fontHeight: 10)
createStyle(name: 'numberData', parent: 'data', h_alignment: 'RIGHT')
createStyle(name: 'dateData', parent: 'data', h_alignment: 'RIGHT', format: 'm/d/yy')
createStyle(name: 'statusGroupTitle', parent: 'header', h_span: 13, h_alignment: 'RIGHT', fontBold: true)
createStyle(name: 'statusGroupNumber', parent: 'statusGroupTitle', h_span: 1)
createStyle(name: 'subagencyGroupTitle', parent: 'statusGroupTitle')
createStyle(name: 'subagencyGroupNumber', parent: 'subagencyGroupTitle', h_span: 1)
createStyle(name: 'totalTitle', parent: 'subagencyGroupTitle')
createStyle(name: 'totalNumber', parent: 'totalTitle', h_span: 1)

class ReportEntry {
    String period
    String day
    String license
    String appId
    String restMethod
    String restLogin
    String sabrePcc
    String sabreAgentSign
    String sabreAction
    String failed
    String soapFault
    String exceptionName
    int count
}

enum ReportColumn {
    license(        'License',      20, 'text', {it -> it.license},         {true}),
    period(         'Period',       20, 'text', {it -> it.period},          {p -> Boolean.TRUE == p.splitByMonthAndDays}),
    days(           'Day',          10, 'text', {it -> it.day},             {p -> Boolean.TRUE == p.splitByMonthAndDays}),
    appId(          'Agency',       20, 'text', {it -> it.appId},           {p -> Boolean.TRUE == p.groupByAgency}),
    restMethod(     'Rest method',  20, 'text', {it -> it.restMethod},      {true}),
    restLogin(      'Rest login',   20, 'text', {it -> it.restLogin},       {true}),
    sabrePcc(       'PCC',          10, 'text', {it -> it.sabrePcc},        {true}),
    sabreAgentSign( 'Agent Sign',   10, 'text', {it -> it.sabreAgentSign},  {true}),
    sabreAction(    'Action',       20, 'text', {it -> it.sabreAction},     {true}),
    failed(         'Failed?',      10, 'text', {it -> it.failed},          {true}),
    soapFault(      'Soap fault?',  10, 'text', {it -> it.soapFault},       {true}),
    exceptionName(  'Exception',    20, 'text', {it -> it.exceptionName},   {true}),
    count(          'Count',        10, 'number',{it -> it.count},          {false})

    String header
    int width
    String valueType
    Closure getter
    Closure grouped

    ReportColumn(String header, int width, String valueType, Closure getter, Closure grouped){
        this.header = header
        this.width = width
        this.valueType = valueType
        this.getter = getter
        this.grouped = grouped
    }
}

class ReportPeriod {
    String caption
    Date startDate
    Date endDate
}

def periods = []
def splitByMonthAndDays = (Boolean.TRUE == parameters.splitByMonthAndDays)
def needArchive = (Boolean.TRUE == parameters.needArchive)

parameters.startDate = MiscUtil.clearTime(parameters.startDate)
parameters.endDate = MiscUtil.setDayEndTime(parameters.endDate)

if (splitByMonthAndDays) {
    if (parameters.startDate == null){
        throw Xeption.forEndUser("A start date must be specified if the monthly/daily option is selected.");
    }
    Date curDate = MiscUtil.clearTime(parameters.startDate)
    Date endDate = parameters.endDate ? MiscUtil.setDayEndTime(parameters.endDate) : MiscUtil.setDayEndTime(new Date())
    while (curDate.before(endDate)) {
        Date periodStartDate = curDate
        Date periodEndDate = MiscUtil.setDayEndTime(curDate)
        periods.add(new ReportPeriod(startDate: periodStartDate, endDate: periodEndDate, caption: formatMonth(periodStartDate)))

        curDate = MiscUtil.addDaysToDate(curDate, 1)
    }
} else {
    periods.add(new ReportPeriod(startDate: MiscUtil.clearTime(parameters.startDate),
            endDate: MiscUtil.setDayEndTime(parameters.endDate), caption: "all_period"))
}

def entries = new ArrayList<ReportEntry>()

periods.forEach { ReportPeriod period ->
    ProjectionQuery pq = new ProjectionQuery()
    if (period.startDate != null) {
        pq.getCriteria().getCriterions().add(
            SearchCriterion.ge(RedAppCounter.Property.sabreDate.name(), MiscUtil.clearTime(period.startDate)))
    }
    if (period.endDate != null) {
        pq.getCriteria().getCriterions().add(
            SearchCriterion.le(RedAppCounter.Property.sabreDate.name(), MiscUtil.setDayEndTime(period.endDate)))
    }
    Collection<EntityReference<MomLicenceCard>> licenses = parameters.license
    if (licenses != null && !licenses.isEmpty()) {
        pq.getCriteria().getCriterions().add(SearchCriterion.in(
                RedAppCounter.Property.licenseUid.name(), licenses.collect {it.uid}.toArray()))
    }

    pq.projections.add(Projection.group(RedAppCounter.Property.licenseUid.name(), "_licenseUid"))
    pq.projections.add(Projection.group(RedAppCounter.Property.appId.name(), "_appId"))
    pq.projections.add(Projection.group(RedAppCounter.Property.restMethod.name(), "_restMethod"))
    pq.projections.add(Projection.group(RedAppCounter.Property.restLogin.name(), "_restLogin"))
    pq.projections.add(Projection.group(RedAppCounter.Property.sabrePcc.name(), "_sabrePcc"))
    pq.projections.add(Projection.group(RedAppCounter.Property.sabreAgentSign.name(), "_sabreAgentSign"))
    pq.projections.add(Projection.group(RedAppCounter.Property.sabreAction.name(), "_sabreAction"))
    pq.projections.add(Projection.group(RedAppCounter.Property.failed.name(), "_failed"))
    pq.projections.add(Projection.group(RedAppCounter.Property.soapFault.name(), "_soapFault"))
    pq.projections.add(Projection.group(RedAppCounter.Property.exceptionName.name(), "_exceptionName"))
    pq.projections.add(Projection.sum(RedAppCounter.Property.count.name(), "_count"))

    Class<? extends BaseAsset> redAppCounterClass = needArchive ? RedAppCounterArchive.class : RedAppCounter.class
    def data = AssetsStorage.get().search(redAppCounterClass, pq).getData()

    data.each { Map<String, Object> entry ->
        def re = new ReportEntry()
        String licenceUid = entry.get("_licenseUid")
        EntityContainer<MomLicenceCard> licenceCtr = EntityStorage.get().load(MomLicenceCard.class, licenceUid)
        licenseCaption = licenceCtr.entity.toString();
        if (licenceCtr != null && licenceCtr.entity.customerName != null) {
            licenseCaption = licenceCtr.entity.customerName
        }
        re.license = licenseCaption
        re.period = period.caption
        re.day = splitByMonthAndDays ? period.startDate?.getDate() : ''
        re.appId = entry.get("_appId")
        re.restMethod = entry.get("_restMethod")
        re.restLogin = entry.get("_restLogin")
        re.sabrePcc = entry.get("_sabrePcc")
        re.sabreAgentSign = entry.get("_sabreAgentSign")
        re.sabreAction = entry.get("_sabreAction")
        boolean failed = entry.get("_failed") as boolean
        re.failed = failed ? 'Failed' : 'Success'
        re.soapFault = failed ? ((entry.get("_soapFault") as boolean) ? 'SoapFault' : 'Other') : ''
        re.exceptionName = entry.get("_exceptionName")
        re.count = entry.get("_count") as int
        entries.add(re)
    }
}

Collections.sort(entries, { ReportEntry e1, ReportEntry e2 ->
    int res = MiscUtil.compare(e1.period, e2.period)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.day, e2.day)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.license, e2.license)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.appId, e2.appId)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.restMethod, e2.restMethod)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.restLogin, e2.restLogin)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.sabrePcc, e2.sabrePcc)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.sabreAgentSign, e2.sabreAgentSign)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.sabreAction, e2.sabreAction)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.failed, e2.failed)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.soapFault, e2.soapFault)
    if (res != 0) {
        return res
    }
    res = MiscUtil.compare(e1.exceptionName, e2.exceptionName)
    if (res != 0) {
        return res
    }
    return 0
})

page { "Data" } {

    ReportColumn.values().each {rc ->
        columnWidth(rc.width); text(rc.header, 'header')
        if (rc != ReportColumn.values().last()) nextColumn()
    }

    entries.forEach { entry ->
        nextRow()
        ReportColumn.values().each {rc ->
            switch (rc.valueType){
                case 'text'     : text(rc.getter(entry), 'textData'); break
                case 'number'   : number(rc.getter(entry), 'numberData'); break
            }
            if (rc != ReportColumn.values().last()) nextColumn()
        }
    }
}

page { "Report" } {

    def agencies = ""
    if (parameters.license != null && !parameters.license.isEmpty()) {
        if (parameters.license.size() < 3) {
            agencies = TextUtil.join(",", parameters.license)
        }
    }

    def periodInfo = getReportPeriod(parameters.startDate, parameters.endDate)
    def title = "RedApps Counter ${agencies} ${periodInfo}"

    rowAutoHeight()
    text(title, 'title', 0, 0)

    nextRow()
    rowHeight(12)

    nextRow()
    pivotTable {
        source(sheet: "Data", leftTop: [row: 0, column: 0],
                rightBottom: [row   : entries.size(), column: ReportColumn.values().last().ordinal()])
        List<Integer> columns = new ArrayList<>()
        ReportColumn.values().each {rc ->
            if (rc.grouped(parameters)) columns.add(rc.ordinal())
        }
        row(columns.toArray(new Integer[0]))
        sumColumn(ReportColumn.count.ordinal())
    }
    nextRow()
    text("", "fakeText"); columnWidth(100)
}

setDefaultSheet("Report")

@Field def monthCaptionMap = [
        (Calendar.JANUARY)  : "01_Jan",
        (Calendar.FEBRUARY) : "02_Feb",
        (Calendar.MARCH)    : "03_Mar",
        (Calendar.APRIL)    : "04_Apr",
        (Calendar.MAY)      : "05_May",
        (Calendar.JUNE)     : "06_Jun",
        (Calendar.JULY)     : "07_Jul",
        (Calendar.AUGUST)   : "08_Aug",
        (Calendar.SEPTEMBER): "09_Sep",
        (Calendar.OCTOBER)  : "10_Oct",
        (Calendar.NOVEMBER) : "11_Nov",
        (Calendar.DECEMBER) : "12_Dec"]

String formatMonth(Date date){
    Calendar cal = Calendar.getInstance()
    cal.setTime(date)
    String month = monthCaptionMap.get(cal.get(Calendar.MONTH))
    return "${cal.get(Calendar.YEAR)}_${month}"
}

def getReportPeriod(Date periodBegin, Date periodEnd) {
    if (periodBegin == null && periodEnd == null) {
        return ""
    }
    return "for the period ${getPeriodDateCaption(periodBegin)} - ${getPeriodDateCaption(periodEnd)}"
}

String getPeriodDateCaption(Date date) {
    return (date == null) ? "..." : DF.format(date)
}