//file:noinspection GrDeprecatedAPIUsage
package com.gridnine.xtrip.server.ibecorp.scripts

import com.gridnine.xtrip.common.model.dict.CodeSystem
import com.gridnine.xtrip.common.model.dict.GeoLocation
import com.gridnine.xtrip.common.model.dict.LocationType
import com.gridnine.xtrip.common.util.CollectionUtil
import com.gridnine.xtrip.common.util.XmlUtil
import com.gridnine.xtrip.common.xml.DocumentBuilderHelper
import com.gridnine.xtrip.common.xml.XUtil
import com.gridnine.xtrip.server.storage.DictionaryStorage
import org.w3c.dom.Element

import java.time.ZoneId

import static com.gridnine.xtrip.server.model.tasks.standard.ExecuteGroovyScriptTask.InputFile

InputFile locationsFile = inputFile
if (locationsFile == null) {
    message("no file, exiting...")
    return
}

def locations  = (Collection<GeoLocation>) DictionaryStorage.get().loadAll(new Date(0), GeoLocation.class)
def deleteCrtCodeLocations = new HashSet<String>()
def addTimeZoneIdLocations = new HashSet<String>()
def crtCode2TimeZoneId = new HashMap<String, String>()
def iataCode2TimeZoneId = new HashMap<String, String>()

buildCodeMaps(locationsFile, crtCode2TimeZoneId, iataCode2TimeZoneId)

for (GeoLocation location : locations) {
    def needsSave = false
    def codeVariants = location.getCodeVariants()
    def crtCode = location.codeVariants.get(CodeSystem.CRT.name())
    def iataCode = location.codeVariants.get(CodeSystem.IATA.name())
    if (codeVariants.size() == 2
            && codeVariants.containsKey(CodeSystem.HOTEL_AGGREGATOR.name())
            && codeVariants.containsKey(CodeSystem.CRT.name())
            && !crtCode2TimeZoneId.containsKey(crtCode)) {
        codeVariants.remove(CodeSystem.CRT.name())
        deleteCrtCodeLocations.add(
                CollectionUtil.firstNonNull(location.getCodeVariants().values()))
        needsSave = true
    }
    if (location.getType() == LocationType.CITY && location.getTimeZoneId() == null) {
        if (crtCode2TimeZoneId.containsKey(crtCode)) {
            location.setTimeZoneId(crtCode2TimeZoneId.get(crtCode))
            addTimeZoneIdLocations.add(
                    CollectionUtil.firstNonNull(location.getCodeVariants().values()))
            needsSave = true
        } else if (iataCode2TimeZoneId.containsKey(iataCode)) {
            location.setTimeZoneId(iataCode2TimeZoneId.get(iataCode))
            addTimeZoneIdLocations.add(
                    CollectionUtil.firstNonNull(location.getCodeVariants().values()))
            needsSave = true
        }
    }
    if (needsSave) {
        location.setDataSource('CleanupGeoLocationCrtCodes')
        DictionaryStorage.get().save(location)
    }
}
StringJoiner joiner = new StringJoiner(" ")
deleteCrtCodeLocations.forEach(joiner.&add)
message(String.format("deleteCrtCodeLocations: %s", joiner.toString()))

joiner = new StringJoiner(" ")
addTimeZoneIdLocations.forEach(joiner.&add)
message(String.format("addTimeZoneIdLocations: %s", joiner.toString()))

static Element getRootElementFromXml(final InputFile xmlData) {
    return new ByteArrayInputStream(xmlData.getBody()).withCloseable {
        stream ->
            DocumentBuilderHelper.parse(XUtil.createSource(stream))
                    .getDocumentElement()
    }
}

static void buildCodeMaps(InputFile xmlFile,
                          Map<String, String> crtCode2TimeZoneId,
                          Map<String, String> iataCode2TimeZoneId) {
    def rootElm = getRootElementFromXml(xmlFile)
    def answerElm = XmlUtil.getElement(rootElm, "answer")
    def describeElm = XmlUtil.getElement(answerElm, "describe")
    def dataElms = XmlUtil.getElements(describeElm, "data")
    def zoneIds = ZoneId.getAvailableZoneIds()

    dataElms.each { data ->
        def codes = XmlUtil.getElements(data, "code")
        def timezone = XmlUtil.getElement(data, "timezone")?.getAttribute("id")
        if (timezone != null) {
            timezone = zoneIds.stream()
                    .filter({ zoneId -> zoneId.equalsIgnoreCase(timezone) })
                    .findAny().orElse(null)
        }
        codes.each { code ->
            if ("ru" == code.getAttribute("xml:lang")) {
                crtCode2TimeZoneId.put(code.getTextContent(), timezone)
            }
            if ("en" == code.getAttribute("xml:lang")) {
                iataCode2TimeZoneId.put(code.getTextContent(), timezone)
            }
        }
    }
}
