package com.gridnine.xtrip.server.ibecorp.tasks.groovy

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.l10n.model.L10nString
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.profile.*
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.search.SearchResult
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String LAST_NAME = "lastName"
@Field String FIRST_NAME = "firstName"
@Field String MIDDLE_NAME = "middleName"
@Field String SECOND_NAME = "secondName"

@Field String EN_LANGUAGE = "en"
@Field String RU_LANGUAGE = "ru"

@Field String TAB = "\t"
@Field String NEW_LINE = "\n"
@Field String ARROW = "->"
@Field String SPACE = " "
@Field String EMPTY = ""

@Field String FILE_SUFFIX = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date())
@Field File FILE = new File(Environment.getTempFolder(), "wrongRuEnLocaleProfiles_${FILE_SUFFIX}.txt")
@Field FileWriter WRITER = new FileWriter(FILE)
@Field FastDateFormat DTF = FastDateFormat.getInstance("yyyy-MM-dd HH:mm:ss")

doJob()

protected void doJob() throws Exception {
    try {
        SearchResult<PersonIndex> searchResult = EntityStorage.get().search(PersonIndex.class, buildQuery())
        for (PersonIndex index : searchResult.getData()) {
            EntityReference<Person> personRef = index.getSource()
            if (personRef != null) {
                EntityContainer<Person> personEtc = EntityStorage.get().resolve(personRef)
                if (personEtc != null) {
                    Person person = personEtc.getEntity()
                    String personCtrUID = personEtc.getUid();
                    if (hasWrongData(person)) {
                        fileMessage(person, personCtrUID)
                    }
                }
            }
        }
    } finally {
        WRITER.close()
    }
}

private static SearchQuery buildQuery() {
    SearchQuery query = new SearchQuery()
    query.getPreferredProperties().add("uid")
    return query
}

private boolean hasWrongData(Person person) {
    if (checkNameForWrongData(person.getLastName())) {
        return true
    }
    if (checkNameForWrongData(person.getFirstName())) {
        return true
    }
    if (checkNameForWrongData(person.getMiddleName())) {
        return true
    }
    if (checkNameForWrongData(person.getSecondName())) {
        return true
    }
    return false
}

private boolean checkNameForWrongData(L10nString name) {
    return name.getValues().entrySet().stream()
            .anyMatch({ entry -> isWrongLocale(entry) })
}

private boolean isWrongLocale(Map.Entry<Locale, String> entry) {
    Locale key = entry.getKey()
    String value = entry.getValue()
    if (key != null) {
        String language = key.getLanguage()
        if (language == EN_LANGUAGE) {
            return isNameContainsWrongData(value, 'А' as char, 'я' as char, true)
        }
        if (language == RU_LANGUAGE) {
            return isNameContainsWrongData(value, 'A' as char, 'z' as char, false)
        }
    }
    return false
}

private static boolean isNameContainsWrongData(String value, char start, char end, boolean withYo) {
    for (int i = 0; i < value.length(); i++) {
        char c = value.charAt(i)
        if ((c >= start && c <= end) || (withYo && (c == 'Ё' as char || c == 'ё' as char))) {
            return true
        }
    }
    return false
}

private void fileMessage(Person person, String personCtrUID) {
    WRITER.write(DTF.format(new Date()))
    WRITER.write(SPACE)
    WRITER.write("Wrong localization found:")
    WRITER.write(NEW_LINE)
    WRITER.write("Profile = ")
    writeFullName(person)
    WRITER.write(NEW_LINE)
    WRITER.write("Ctr UID = ")
    WRITER.write(personCtrUID)
    WRITER.write(NEW_LINE)
    WRITER.write("Employment = ")
    writeEmployment(person)
    WRITER.write(NEW_LINE)
    WRITER.write("Data: [")
    WRITER.write(NEW_LINE)
    writeFullWrongData(person)
    WRITER.write("]")
    WRITER.write(NEW_LINE)
}

private void writeEmployment(Person person) {
    List<String> employments = new ArrayList<>()
    for (PersonEmployment pe : person.getEmployments()) {
        if (pe.getOrganization() == null || pe.getOrganization().caption == null) {
            continue
        }
        employments.add(pe.getOrganization().caption)
    }
    String employment = String.join(", ", employments);
    WRITER.write(employment);
}

private void writeFullName(Person person) {
    writeName(person.getLastName(), SPACE)
    writeName(person.getFirstName(), SPACE)
    writeName(person.getMiddleName(), SPACE)
    writeName(person.getSecondName(), EMPTY)
}

private void writeName(L10nString names, String postfix) {
    for (Map.Entry<Locale, String> entry : names.getValues().entrySet()) {
        WRITER.write(entry.getValue())
        WRITER.write(postfix)
        break
    }
}

private void writeFullWrongData(Person person) {
    writeWrongData(person.getLastName(), LAST_NAME)
    writeWrongData(person.getFirstName(), FIRST_NAME)
    writeWrongData(person.getMiddleName(), MIDDLE_NAME)
    writeWrongData(person.getSecondName(), SECOND_NAME)
}

private void writeWrongData(L10nString names, String field) {
    names.getValues().entrySet().stream()
            .filter({ entry -> isWrongLocale(entry) })
            .forEach({ entry ->
                WRITER.write(TAB)
                WRITER.write(field)
                WRITER.write(SPACE)
                WRITER.write(ARROW)
                WRITER.write(SPACE)
                WRITER.write(entry.getKey().getLanguage())
                WRITER.write(SPACE)
                WRITER.write(ARROW)
                WRITER.write(SPACE)
                WRITER.write(entry.getValue())
                WRITER.write(NEW_LINE)
            })
}