package com.gridnine.xtrip.server.ibecorp.tasks.groovy.xtr13665

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.PassengerPassportIndex
import com.gridnine.xtrip.common.model.dict.PassportType
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.profile.Organization
import com.gridnine.xtrip.common.model.profile.Person
import com.gridnine.xtrip.common.model.profile.PersonPassportWrapper
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.search.SortOrder
import com.gridnine.xtrip.common.search.utils.SearchQueryHelper
import com.gridnine.xtrip.common.util.CollectionUtil
import com.gridnine.xtrip.common.util.TextUtil
import com.gridnine.xtrip.common.util.TranslitUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date());
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm");

@Field boolean onlyCount = true;
@Field boolean stopWhenError = true;
@Field int limit = 0;
@Field int updateLimit = 13000;

@Field List<EntityReference<Organization>> orgRefs = null;
//@Field List<EntityReference<Organization>> orgRefs = Arrays.asList(
//        new EntityReference<>("8b280a2d-6a22-4415-8571-86ac33f36be0", Organization.class, null),
//        new EntityReference<>("83779a60-0f85-4feb-a77c-af9de5a64da4", Organization.class, null),
//        new EntityReference<>("18ab1640-89d2-4e4a-9363-88ac91bd42b0", Organization.class, null),
//        new EntityReference<>("b799ec06-7d43-4d9c-8bd3-bbb81e86f5c7", Organization.class, null),
//        new EntityReference<>("7790b178-42e5-4078-82df-6841cfd75e69", Organization.class, null),
//        new EntityReference<>("1badf562-0256-439e-a088-4a61bb75b01b", Organization.class, null),
//        new EntityReference<>("60534773-9d97-4d09-8946-938b64bb5eba", Organization.class, null),
//        new EntityReference<>("c4fba469-e4c0-488c-8e4d-edcb9021cac1", Organization.class, null),
//        new EntityReference<>("646314bc-8f8d-4cdc-aa31-b8c853d05b54", Organization.class, null),
//        new EntityReference<>("7ba0c653-b914-4f69-a4d2-66058bf348e7", Organization.class, null),
//)

if (onlyCount){
    fileMessage("Режим 'Только подсчет'");
}
doJob()


void doJob(){
    fileMessage("Останавливать процесс при ошибке? ${stopWhenError ? 'Да' : 'Нет'}");
    fileMessage("");

    SearchQuery query = new SearchQuery();
    query.getCriteria().getCriterions().add(
            SearchCriterion.eq(PassengerPassportIndex.Property.passportType.name(), PassportType.INTERNAL));

    if (CollectionUtil.isNotEmpty(orgRefs)) {
        List<String> orgNames = new ArrayList<>()
        orgRefs.each {
            EntityContainer<Organization> orgCtr = EntityStorage.get().resolve(it);
            if (orgCtr == null) {
                throw Xeption.forEndUser("Невалидная ссылка на организацию {0}", orgRef.getUid());
            }
            orgNames.add(orgCtr.getEntity().toString())
        }
        fileMessage("Фильтр по организациям ${orgNames}");
        query.getCriteria().getCriterions().add(
                SearchQueryHelper.buildOrEqEntityCriterion(PassengerPassportIndex.Property.company.name(), orgRefs))
    }

    query.getPreferredProperties().add("containerUid");
    query.getPreferredProperties().add(PassengerPassportIndex.Property.company.name());
    query.getCriteria().getOrders().put("companyReferenceCaption", SortOrder.ASC)
    if (!onlyCount){
        if (limit > 0) {
            query.setLimit(limit)
            fileMessage("Установлено ограничение на поиск $limit");
        }
        fileMessage("Установлено ограничение на изменение $updateLimit");
    }

    List<PassengerPassportIndex> pspIndices = EntityStorage.get().search(PassengerPassportIndex.class, query).getData();

    fileMessage("Найдено ${pspIndices.size()} паспортов РФ");

    if (pspIndices.isEmpty()) {
        return;
    }

    pspIndices = pspIndices.unique({ a, b -> a.source?.uid <=> b.source?.uid })

    def total = pspIndices.size()
    def current = 0
    def progr = 0
    int errors = 0
    int changed = 0;
    int unChanged = 0;


    fileMessage("А именно ${total} пассажиров с паспортами РФ");
    fileMessage("");
    fileMessage("");

    Map<EntityReference, List<PassengerPassportIndex>> groupedIndices = pspIndices.groupBy { it.company }

    groupedIndices.each {oRef, list ->
        if (oRef==null){
            fileMessage("Обработка ${list.size()} пассажиров без организации");
        } else {
            fileMessage("Обработка ${list.size()} пассажиров организации ${oRef.caption} (${oRef.uid})");
        }
        List<String> changedPersons = new ArrayList<>();
        List<String> unChangedPersons = new ArrayList<>();
        int errorsByOrg = 0;
        for (PassengerPassportIndex ind : list) {
            if (isToBeStopped()) {
                fileMessage("iterator stopped")
                break;
            }
            if (changed > updateLimit){
                fileMessage("Обработка завершена в соответствии с лимитом ${updateLimit}")
                break;
            }
            EntityReference<Person> ref =  ind.source
            try {
                EntityContainer<Person> personCtr = EntityStorage.get().resolve(ref);
                if (updated(personCtr.getEntity())){
                    if (!onlyCount){
                        personCtr.getVersionInfo().setDataSource("XTR-13665");
                        personCtr.getVersionInfo().setVersionNotes("Транслитирация ФИО (заполенние локали EN) в паспорте РФ");
                        EntityStorage.get().save(personCtr, true);
                    }
                    changed++
                    changedPersons.add("{${personCtr.getEntity().toString()} (${personCtr.toReference().getUid()})}");
                } else {
                    unChanged++
                    unChangedPersons.add("{${personCtr.getEntity().toString()} (${personCtr.toReference().getUid()})}");
                }
            } catch (Exception ex) {
                errorsByOrg++
                errors++;
                fileMessage("Ошибка (${ind.source.caption} / ${ind.source.uid}) : ${TextUtil.getExceptionStackTrace(ex)}", false);
                if (stopWhenError){
                    fileMessage("Процесс остановлен из-за ошибки в профиле ${ref.caption} / ${ref.uid}");
                    throw new Exception("Произошла ошибка при обработке пассажира", ex);
                }
            }

            def progrNew = Math.round(current * 100d / total)
            if (progr != progrNew) {
                progress("${current + 1} / ${total}".toString(), (int) progrNew)
            }
            progr = progrNew
            current++
        }
        fileMessage("Изменено ${changedPersons.size()}, Не изменено ${unChangedPersons.size()}, Ошибок ${errorsByOrg}");
        fileMessage("Список измененных пассажиров : ${changedPersons.toString()}", false);
        //fileMessage("Список неизмененных пассажиров : ${unChangedPersons.toString()}", false);
        fileMessage("");
        fileMessage("");
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("Изменено пассажиров: ${changed}");
    fileMessage("Не изменено пассажиров: ${unChanged}");
    fileMessage("Возникло ошибок: ${errors}");
}

boolean updated(Person person){
    List<PersonPassportWrapper> passports = PersonPassportWrapper.wrap(person.passports)
    boolean updated = false;
    for (PersonPassportWrapper pspWr : passports){
        if (pspWr.getType() != PassportType.INTERNAL) {
            continue;
        }
        if (TextUtil.isBlank(pspWr.getLastNameLatin()) && TextUtil.nonBlank(pspWr.getLastNameCyrillic())){
            pspWr.setLastNameLatin(cyr2lat(pspWr.getLastNameCyrillic()))
            updated = true
        }
        if (TextUtil.isBlank(pspWr.getFirstNameLatin()) && TextUtil.nonBlank(pspWr.getFirstNameCyrillic())){
            pspWr.setFirstNameLatin(cyr2lat(pspWr.getFirstNameCyrillic()))
            updated = true
        }
        if (TextUtil.isBlank(pspWr.getMiddleNameLatin()) && TextUtil.nonBlank(pspWr.getMiddleNameCyrillic())){
            pspWr.setMiddleNameLatin(cyr2lat(pspWr.getMiddleNameCyrillic()))
            updated = true
        }
        if (TextUtil.isBlank(pspWr.getSecondNameLatin()) && TextUtil.nonBlank(pspWr.getSecondNameCyrillic())){
            pspWr.setSecondNameLatin(cyr2lat(pspWr.getSecondNameCyrillic()))
            updated = true
        }
    }
    if (updated && !onlyCount){
        person.passports.clear();
        passports.each {wr -> wr.unwrap(person.passports)}
    }
    return updated
}

String cyr2lat(String value) {
    return TranslitUtil.cyr2lat(value, TranslitUtil.TranslitRules.GOST_52535_1_2006);
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str);
    }
    String suffix = onlyCount ? "-onlyCount" : ""
    new File(Environment.getTempFolder(), "xtr13665${suffix}-${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n");
}