package com.gridnine.xtrip.server.model.helpers.vat.air

import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.booking.BaseProduct
import com.gridnine.xtrip.common.model.booking.PassengerType
import com.gridnine.xtrip.common.model.booking.VatBasisType
import com.gridnine.xtrip.common.model.booking.VatComponent
import com.gridnine.xtrip.common.model.booking.VatDetalization
import com.gridnine.xtrip.common.model.dict.*
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.handlers.ProductHandler
import com.gridnine.xtrip.common.model.helpers.AirProductHelper
import com.gridnine.xtrip.common.model.helpers.AirProductTaxHelper
import com.gridnine.xtrip.common.model.helpers.DictHelper
import com.gridnine.xtrip.common.model.helpers.ProfileHelper
import com.gridnine.xtrip.common.model.helpers.SystemHelper
import com.gridnine.xtrip.common.model.profile.Organization
import com.gridnine.xtrip.common.model.validation.StandartValidationMessageType
import com.gridnine.xtrip.common.model.validation.ValidationMessageHelper
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import groovy.transform.Field

import java.math.RoundingMode

class TaxesVatCalculationParams {
    final Collection<String> taxCodes

    final boolean excludeTaxesCodes

    TaxesVatCalculationParams(final Collection<String> taxCodes,
                              final boolean excludeTaxesCodes) {
        this.taxCodes = taxCodes
        this.excludeTaxesCodes = excludeTaxesCodes
    }

}

enum TransportationType{
    MVL,
    VVL,
    ZVL,
    ZERO_VVL,
    UNDEFINED  
}

private static<P, T, ST, S> void updateVatDetalization(VatDetalization detalization, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    DictionaryReference<Airline> validatingCarrier = adapter.getValidatingAirline(product)
    //XTR-6898
    if(isForeignAirline(validatingCarrier)){
        if(isTCHTicket(product, adapter)){
            //FOR MCO for Prepayment for Group Tickets VAT IS NULL
            if (adapter.getMcoCategory(product) == MCOCategory.GROUP_PREPAYMENT) {
                return
            }
            //VAT FOR ZZ TAXES IS ADDED IN ANY WAY WITH RATE 18
            detalization.getComponents()
                    .addAll(createZZVatComponents(product,passengerType, adapter))
            return
        }
        return
    }
    // XTRKZ-2
    //VIP-27588
    if (SystemHelper.isKazakhInstallation()) {
        updateKzVAT(detalization, product, passengerType, adapter)
        return
    }
    //на iata код ориентироваться во всех случаях нельзя из за Аэрофлота у ВИПа
    //у них много аэрофлотов с разными кодами. их объединяет то что в коде есть 555
    String code = DictHelper.getCodeVariant(validatingCarrier, CodeSystem.IATA)
    if(code == null){
        code = validatingCarrier == null ? null
                : validatingCarrier.getCode()
    }

    //XTR-9459
    if(("N4" == code || "КЛ" == code) &&
//            XTR-10859
            ProductCategory.MCO != ProductHandler.of(product).getProductCategory(product)){
        updateN4VAT(detalization, product,  passengerType, adapter)
        return
    }
    //XTR-8865
    if ((code != null) && ("SU" == code || (validatingCarrier!= null && validatingCarrier.getCode().contains("555")) )) {
        updateSuVAT(detalization, product, passengerType,adapter)
        return
    }

    //XTR-3782
    if(isTCHTicket(product, adapter)){
        updateTchVAT(detalization, product, passengerType, adapter)
        return
    }

    //XTR-3812
    if ("UN" == code) {
        updateUnVAT(detalization, product,passengerType,adapter)
        return
    }
    //VIP-12926
    //XTR-8665
    //VIP-47430
    if ("UT" == code) { 
        updateUtVAT(detalization, product, passengerType,adapter)
        return
    }
    if(matchesKZVatAlgorithm(product, adapter)){
        updateKzVAT(detalization, product, passengerType, adapter)
        return
    }
    //XTR-4067
    if ("S7" == code) {
        updateS7VAT(detalization, product, passengerType,adapter)
        return
    }
    //XTR-6019
    //VIP-41292
    if ("U6" == code) { 
        updateU6VAT(detalization, product, passengerType,adapter)
        return
    }
    //VIP-37045
    //в некоторых базах ДР(XTRIP) в некоторых DP (VIP).
    if ("DP" == code || "ДР" == code) { //$NON-NLS-1$
        updateDpVAT(detalization, product, passengerType,adapter)
        return
    }
    //XTR-3817
    updateStandardVAT(detalization, product, passengerType, adapter)
}
/*
XTR-3817
*/
private static<P, T, ST, S> void updateStandardVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (mcoCategory == MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    result.getComponents()
            .addAll(createZZVatComponents(product,passengerType, adapter))
    //VAT IS CALCULATED ONLY FOR AVIATICKETS AND EXCESS_BAGAGE
    ProductCategory productCategory = adapter.getProductCategory(product)
    if (productCategory != ProductCategory.AIR && productCategory != ProductCategory.EXCESS_BAGAGE && adapter.getMcoCategory(product) != MCOCategory.EXCESS_LUGGAGE) {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        taxesExceptZZ, false, false, passengerType, adapter))
        return
    }
    DictionaryReference<Airline> validatingAirline = adapter.getValidatingAirline(product)
    String code = DictHelper.getCodeVariant(validatingAirline, CodeSystem.IATA)
    if(code == null){
        code = validatingAirline == null ? null
                : validatingAirline.getCode()
    }
    boolean calculateRItax = code == null || !noRICarriers.contains(code)
    if (isHasVat(product,adapter)) {
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product,adapter), true,
                        calculateRItax? defaultStandardTaxesWithRIVatCalculationParams: defaultStandardTaxesVatCalculationParams, true, false, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        calculateRItax? defaultRestTaxesWithoutRIVatCalculationParams: defaultRestTaxesVatCalculationParams, false, false, passengerType, adapter))
        return
    }
    result.getComponents()
            .addAll(createComponents(product,0, true,
                    taxesExceptZZ, false,false, passengerType, adapter))
}
/*
DP
VIP-37045
 */
private static<P, T, ST, S> void updateDpVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    result.getComponents()
            .addAll(createZZVatComponents(product,passengerType,adapter))
    //VAT IS CALCULATED ONLY FOR AVIATICKETS AND EXCESS_EXCESS_BAGAGE as in DefaultVatCalculator
    ProductCategory productCategory = adapter.getProductCategory(product)
    if (productCategory != ProductCategory.AIR && productCategory != ProductCategory.EXCESS_BAGAGE && adapter.getMcoCategory(product) != MCOCategory.EXCESS_LUGGAGE) {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        taxesExceptZZ, false, false, passengerType, adapter))
        return
    }
    if (isHasVat(product,adapter)) {
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product,adapter), true,
                        dpStandardTaxesVatCalculationParams, true, false, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,
                        CommonAirProductVatHelper.getStandardVATRate(product,adapter), false,
                        yrTaxCalculationParams, false, false, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product,adapter), false,
                        dpB110TaxesCalculationParams, true, false, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,
                        getStandardVATRate(product,adapter), false,
                        dpSMSTaxesCalculationParams, true, false, passengerType, adapter));
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        dpRestTaxesVatCalculationParams, false, false, passengerType, adapter))
        return
    }
    result.getComponents()
            .addAll(createComponents(product,0, true,
                    taxesExceptZZ, false, false, passengerType, adapter))
}
/*
U6
XTR-6019
VIP-41292
 */
private static<P, T, ST, S> void updateU6VAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (mcoCategory == MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    result.getComponents()
            .addAll(createZZVatComponents(product, passengerType,adapter))
    //VAT IS CALCULATED ONLY FOR AVIATICKETS
    ProductCategory productCategory = adapter.getProductCategory(product)
    if (productCategory != ProductCategory.AIR) {
        //VIP-41292
        if(productCategory == ProductCategory.EXCESS_BAGAGE || mcoCategory == MCOCategory.EXCESS_LUGGAGE){
            if (isHasVat(product, adapter)) {
                result.getComponents()
                        .addAll(createComponents(product,
                                getReducedVATRate(product,adapter), true,
                                fareComponent, false, false, passengerType,adapter))
                result.getComponents()
                        .addAll(createComponents(product,0, false,
                                taxesExceptZZ, false,false, passengerType,adapter))
                return
            }
        }
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        taxesExceptZZ, false, false, passengerType,adapter))
        return
    }
    if (isHasVat(product,adapter)) {
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product,adapter), true,
                        taxesExceptZZ, false, false, passengerType,adapter))
    } else {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        taxesExceptZZ, false, false, passengerType,adapter))
    }
}
/*S7
XTR-4067
 */
private static<P, T, ST, S> void updateS7VAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (mcoCategory == MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    result.getComponents()
            .addAll(createZZVatComponents(product, passengerType,adapter))
    ProductCategory productCategory = adapter.getProductCategory(product)
    if (productCategory == null) {
        adapter.addValidationMessage(product, ValidationMessageHelper.createValidationMessage(
                StandartValidationMessageType.APVC_NO_PRODUCT_CATEGORY))
        result.getComponents()
                .addAll(createComponents(
                        product, 0,
                        true, s7TaxComponents, false, false, passengerType,adapter))
        return
    }
    switch (productCategory) {
        case ProductCategory.MCO:
            if (mcoCategory == null) {
                adapter.addValidationMessage(product,
                        ValidationMessageHelper.createValidationMessage(
                                StandartValidationMessageType.APVC_NO_PRODUCT_CATEGORY))
                result.getComponents()
                        .addAll(createComponents(
                                product, 0,
                                true, s7TaxComponents, false,false, passengerType, adapter))
                break
            }
            switch (mcoCategory) {
                case MCOCategory.EXCESS_LUGGAGE:
                    updateS7FareComponent(result, product,passengerType,adapter)
                    break
                case MCOCategory.SPECIAL_FOOD:
                case MCOCategory.SEAT_RESERVATION:
                    updateS7FareComponent(result, product,passengerType,adapter)
                    break
                default:
                    result.getComponents()
                            .addAll(createComponents(
                                    product, 0,
                                    true, s7TaxComponents, false, false, passengerType,adapter))
            }
            break
        case ProductCategory.EXCESS_BAGAGE:
            updateS7FareComponent(result, product,passengerType,adapter)
            break
        default:
            if (isHasVat(product,adapter)) {
                result.getComponents()
                        .addAll(createComponents(
                                product, getReducedVATRate(product, adapter),
                                true, s7TaxComponents, false, false, passengerType,adapter))
            } else{
                result.getComponents()
                        .addAll(createComponents(
                                product, 0,
                                true, s7TaxComponents, false, false, passengerType,adapter))
            }
    }
}

private static<P, T, ST, S> void updateS7FareComponent(final VatDetalization result,
                                                     final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    if (!isHasVat(product, adapter)) {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        fareComponent, false, false, passengerType,adapter))
        return
    }
    result.getComponents()
            .addAll(createComponents(product,
                    getReducedVATRate(product,adapter), true,
                    fareComponent, false, false, passengerType,adapter))
}
/* UT
    VIP-12926
    XTR-8665
    VIP-47430
 */
private static<P, T, ST, S> void updateUtVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (mcoCategory == MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    result.getComponents()
            .addAll(createZZVatComponents(product, passengerType, adapter))
    //VAT IS CALCULATED ONLY FOR AVIATICKETS AND MCO EXSESS_LUGGAGE
    ProductCategory productCategory = adapter.getProductCategory(product)
    if (productCategory != ProductCategory.AIR && productCategory != ProductCategory.EXCESS_BAGAGE&& mcoCategory != MCOCategory.EXCESS_LUGGAGE) {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        utTaxComponent,
                        true, false, passengerType, adapter))
        return
    }
    if (isHasVat(product, adapter)) {
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product, adapter), true,
                        utTaxComponent,
                        true,false, passengerType, adapter))
    } else {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        utTaxComponent,
                        true, false, passengerType,adapter))
    }
}
/* UN
XTR-3812
 */
private static<P, T, ST, S> void updateUnVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (mcoCategory== MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    result.getComponents()
            .addAll(createZZVatComponents(product,passengerType, adapter))
    //VAT IS CALCULATED ONLY FOR AVIATICKETS AND MCO EXSESS_LUGGAGE
    ProductCategory productCategory = adapter.getProductCategory(product)
    if (productCategory != ProductCategory.AIR && productCategory != ProductCategory.EXCESS_BAGAGE&& mcoCategory != MCOCategory.EXCESS_LUGGAGE) {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        taxesExceptZZ,
                        true, false, passengerType, adapter))
        return
    }
    if (isHasVat(product, adapter)) {
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product,adapter), true,
                        taxesExceptZZ, 
                        true, false, passengerType,adapter))
    } else {
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        taxesExceptZZ,
                        true, false, passengerType,adapter))
    }
}
/*TCH
    XTR-3782
 */
private static<P, T, ST, S> void updateTchVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    if (adapter.getMcoCategory(product) == MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    //VAT FOR ZZ TAXES IS ADDED IN ANY WAY WITH RATE 18
    result.getComponents()
            .addAll(createZZVatComponents(product,passengerType, adapter))
    //excess laggage
    if ((adapter.getMcoCategory(product) == MCOCategory.EXCESS_LUGGAGE)
            || (adapter.getProductCategory(product) == ProductCategory.EXCESS_BAGAGE)) {
        if (isHasVat(product, adapter)) {
            result.getComponents()
                    .addAll(createComponents(product,
                            getReducedVATRate(product, adapter),
                            true, fareComponent, true, false, passengerType, adapter))
        } else{
            result.getComponents()
                    .addAll(createComponents(product,0,
                            true, fareComponent, true, false, passengerType,adapter))
        }
        return
    }
    //other mco
    if (adapter.getProductCategory(product) == ProductCategory.MCO) {
        result.getComponents()
                .addAll(createComponents(product,0,
                        true, fareComponent, true,  false, passengerType,adapter))
        addTchTaxesComponents(result, product, 0, passengerType,adapter)
        return
    }
    Set<String> airlinesCodes = new HashSet<>()
    for (ST st : adapter.getSegmentTariffs(product)) {
        for (S seg : adapter.getSegments(st)) {
            DictionaryReference<Airline> marketingAirline = adapter.getMarketingAirline(seg)
            if (marketingAirline != null) {
                airlinesCodes.add(marketingAirline.getCode())
            }
        }
    }
    //XTR-6898
    if (airlinesCodes.size() > 1) {
        result.getComponents()
                .addAll(createComponents(product,0,
                        true, fareComponent, true, false, passengerType,adapter))
        addTchTaxesComponents(result, product, 0,passengerType, adapter)
        return
    }
    if (isHasVat(product,adapter)) {
        result.getComponents()
                .addAll(createComponents(product,
                        getReducedVATRate(product,adapter),
                        true, fareComponent, true, false, passengerType,adapter))
        addTchTaxesComponents(result, product, getReducedVATRate(product,adapter), passengerType, adapter)
    } else{
        result.getComponents()
                .addAll(createComponents(product,0,
                        true, fareComponent, true, false, passengerType,adapter))
        addTchTaxesComponents(result, product, 0, passengerType, adapter)
    }
}

private static<P, T, ST,S> void addTchTaxesComponents(final VatDetalization result, P product, double rate, DictionaryReference<PassengerType> passengerType,AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    DictionaryReference<Airline> validatingAirline = adapter.getValidatingAirline(product)
    String code = DictHelper.getCodeVariant(validatingAirline, CodeSystem.IATA)
    if(code == null){
        code = validatingAirline == null ? null
                : validatingAirline.getCode()
    }
    boolean includeRITax = validatingAirline != null && !noRICarriers.contains(code)
    for (TaxesVatCalculationParams taxComp : includeRITax? tchReduceVATTaxesWithRIComps: tchReduceVATTaxesComps) {
        result.getComponents()
                .addAll(
                        createComponents(product,rate, false,
                                taxComp, true, true, passengerType,adapter))
    }
    if ("DP" == code || "ДР" == code) {
        rate = CommonAirProductVatHelper.getStandardVATRate(product,adapter)
    }
    result.getComponents()
            .addAll(
                    createComponents(product, rate, false,
                            yrTaxCalculationParams, false, false, passengerType, adapter))
}
/*SU
   XTR-8865
 */
private static<P, T, ST, S> void updateSuVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //VAT FOR ZZ TAXES IS ADDED IN ANY WAY WITH RATE 18
    result.getComponents()
            .addAll(createZZVatComponents(product,passengerType,adapter))
    //TAX YR added in any way after 2019-11-11
    result.getComponents().addAll(createComponents(product,
                            getStandardVATRate(product, adapter),
                            false, yrTaxCalculationParams, false, false, passengerType, adapter))
    ProductCategory productCategory = adapter.getProductCategory(product)
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (productCategory!= ProductCategory.AIR) {
        if (mcoCategory == null) {
            return
        }
        TransportationType tt = getTransportationType(adapter.getTransportationTypeProduct(product), adapter)
        switch (mcoCategory) {
            case MCOCategory.GROUP_PREPAYMENT:
                switch (tt) {
                    case TransportationType.ZERO_VVL:
                        result.getComponents()
                                .addAll(createComponents(product, 0, true,
                                                fareComponent, false, false, passengerType,adapter))
                        return
                    default:
                        result.getComponents()
                                .addAll(createComponents(product,
                                                getReducedVATRate(product,adapter),
                                                true, fareComponent, false, false, passengerType,adapter))
                        return

                }
            case MCOCategory.EXCESS_LUGGAGE:
            case MCOCategory.COMFORT_PLUS:
            case MCOCategory.ADDITIONAL_TARIFF:
            case MCOCategory.UNESCORTED_MINOR:
            case MCOCategory.RETURN_TICKET:
            case MCOCategory.SEAT_RESERVATION:
                switch (tt) {
                    case TransportationType.ZERO_VVL:
                    case TransportationType.MVL:
                        result.getComponents()
                                .addAll(createComponents(product, 0, true,
                                                fareComponent, false,false, passengerType,adapter))
                        return
                    case TransportationType.VVL:
                        result.getComponents()
                                .addAll(createComponents(product,
                                                getReducedVATRate(product,adapter),
                                                true, fareComponent, false,false, passengerType,adapter))
                        return

                }
        }
        return
    }
    TransportationType tt =getTransportationType(product, adapter)
    switch (tt) {
        case TransportationType.MVL:
        case TransportationType.ZERO_VVL:
            result.getComponents()
                    .addAll(createComponents(product,
                            0, false, suFuelChargeComponent, false,false, passengerType,adapter))
            result.getComponents().addAll(createComponents(product, 0, true, fareComponent, false, false, passengerType,adapter))
            break
        case TransportationType.VVL:
            result.getComponents()
                    .addAll(createComponents(product,
                            getReducedVATRate(product,adapter),
                            false, suFuelChargeComponent, false, false, passengerType,adapter))
            //VAT FOR FARE IS ADDED ONLY FOR VVL  WITH RATE 10
            result.getComponents()
                    .addAll(createComponents(product,
                            getReducedVATRate(product,adapter),
                            true, fareComponent, false,false, passengerType,adapter))
            break
        case TransportationType.ZVL:
        case TransportationType.UNDEFINED:
            break
    }
}
/* N4
* see history in com.gridnine.xtrip.server.model.helpers.vat.air.N4VatCalculator
* XTR-9712 XTR-9459 XTR-8617 VIP-46744 VIP-46740 XTR-8521
* */
private static<P, T, ST, S> void updateN4VAT(VatDetalization result , final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    result.getComponents()
            .addAll(createZZVatComponents(product,passengerType, adapter))
    Date issueDate = adapter.getIssueDate(product)
    if(issueDate == null || issueDate.before(date_2020_10_01)){
        result.getComponents().addAll(createComponents(product, getStandardVATRate(product, adapter), false, yqTaxCalculationParams,
                true, true, passengerType, adapter))
    }

    result.getComponents().addAll(createComponents(product, getStandardVATRate(product, adapter), false, taxesExceptZZ_RI_YQ_YR_TR_M6_CU_Q5,
            true, true, passengerType, adapter))
    if (!isHasVat(product,adapter)) {
        if(issueDate != null && !issueDate.before(date_2020_10_01) ) {
            result.getComponents().addAll(createComponents(product, 0, false, yqTaxCalculationParams,
                    true, true, passengerType, adapter))
        }
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        riTaxCalculationParams, true,true, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        trTaxCalculationParams, true,true, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        m6TaxCalculationParams, true,true, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        cuTaxCalculationParams, true,true, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,0, false,
                        q5TaxCalculationParams, true,true, passengerType, adapter))
        result.getComponents()
                .addAll(createComponents(product,0, true,
                        yrTaxCalculationParams, true,true, passengerType, adapter))

        return
    }
    if(issueDate != null && !issueDate.before(date_2020_10_01) ) {
        result.getComponents().addAll(createComponents(product, getReducedVATRate(product, adapter), false, yqTaxCalculationParams,
                true, true, passengerType, adapter))
    }
    result.getComponents()
            .addAll(createComponents(product,getReducedVATRate(product, adapter), false,
                    riTaxCalculationParams, true,true, passengerType, adapter))
    result.getComponents()
            .addAll(createComponents(product,getReducedVATRate(product, adapter), false,
                    trTaxCalculationParams, true,true, passengerType, adapter))
    result.getComponents()
            .addAll(createComponents(product,getReducedVATRate(product, adapter), false,
                    m6TaxCalculationParams, true,true, passengerType, adapter))
    result.getComponents()
            .addAll(createComponents(product,getReducedVATRate(product, adapter), false,
                    cuTaxCalculationParams, true,true, passengerType, adapter))
    result.getComponents()
            .addAll(createComponents(product,getReducedVATRate(product, adapter), false,
                    q5TaxCalculationParams, true,true, passengerType, adapter))
    result.getComponents()
            .addAll(createComponents(product,getReducedVATRate(product, adapter), true,
                    yrTaxCalculationParams, true,true, passengerType, adapter))
}

/*KZ
* XTRKZ-2
* VIP-27588
* */
private static<P, T, ST, S> void updateKzVAT(VatDetalization result, final P product, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    //FOR MCO for Prepayment for Group Tickets VAT IS NULL
    MCOCategory mcoCategory = adapter.getMcoCategory(product)
    if (mcoCategory == MCOCategory.GROUP_PREPAYMENT) {
        return
    }
    //VAT IS CALCULATED ONLY FOR AVIATICKETS
    if (isInternal(product, KZ_COUNTRY_CODE,adapter)) {
        result.getComponents().addAll(createComponents(product, 12, true, kzTaxComponent, false,false, passengerType, adapter))
    } else{
        result.getComponents().addAll(createComponents(product, 0, true, kzTaxComponent, false, false, passengerType, adapter))
    }
}
private static<P, T, ST,S> boolean matchesKZVatAlgorithm(final P product, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    Airline carrier =
            DictionaryCache.get().resolveReference(adapter.getValidatingAirline(product))
    boolean isKZcarrier = carrier != null && carrier.getCountry() != null && KZ_COUNTRY_CODE == carrier.getCountry().getCode()
    if (!isKZcarrier) {
        return false
    }
    EntityReference<Organization> supplierRef =
            adapter.getSupplier(product)
    EntityContainer<Organization> supplierCtr =
            EntityStorage.get().resolve(supplierRef)
    if (supplierCtr == null) {
        return false
    }
    DictionaryReference<Country> countryRef =
            ProfileHelper.getOrganizationCountry(supplierCtr.getEntity())
    return countryRef != null && KZ_COUNTRY_CODE == countryRef.getCode()
}
/*Common functions*/
private static <P, T, ST,S> boolean isTCHTicket(P product, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    String blankOwnerNumber = adapter.getBlankOwnerNumber(product)
    return ProfileHelper.isOrganizationCode(adapter.getBlankOwner(product),"Ш1") || "99A" == blankOwnerNumber|| "99А" == blankOwnerNumber
}

private static boolean isForeignAirline(DictionaryReference<Airline> validatingCarrier){
    //VIP-40138
    final Airline validatingAirline =
            DictionaryCache.get().resolveReference(validatingCarrier)
    if (validatingAirline != null) {
        final Country country = DictionaryCache.get()
                .resolveReference(validatingAirline.getCountry())
        if ((country != null) && !country.isDomestic()) {
            return true
        }
    }
    return false
}

private static<P,T,ST,S> double getStandardVATRate(P prod, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    Date vatIssueDate = adapter.getVatIssueDate(prod)
    final BigDecimal vat = DictHelper.getDefaultVat(vatIssueDate)
    double defaultVat = vat != null ? vat.doubleValue() : 18d
    //XTR-7292
    if (SystemHelper.isRussianInstallation()) {
        Date date = vatIssueDate
        if (date != null) {
            Calendar cal = Calendar.getInstance()
            cal.setTime(date)
            int year = cal.get(Calendar.YEAR)
            if ((year >= 2019)) {
                return 20
            }
        }
        return defaultVat
    }
    return defaultVat
}

private static<P,T,ST,S> List<VatComponent> createZZVatComponents(final P prod,  DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    return createComponents(prod, getStandardVATRate(prod, adapter), false, zzTaxComponent,
            false, false, passengerType, adapter)
}

private static BigDecimal calculateVatAmount(final BigDecimal total,
                                            final double rate) {
    return (total * BigDecimal.valueOf(rate)).divide(
            BigDecimal.valueOf(100).add(BigDecimal.valueOf(rate)), 2,
            RoundingMode.HALF_UP)
}

private static<P,T,ST,S> List<VatComponent> createComponents(final P prod,
                                                     final double rate, final boolean includeFare,
                                                     final TaxesVatCalculationParams taxesParams,
                                                     final boolean separateComponents, boolean groupTaxesWithSameCodes, DictionaryReference<PassengerType> passengerType, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    List<VatComponent> result = new ArrayList<VatComponent>()
    BigDecimal totalBasis = BigDecimal.ZERO
    Set<String> taxesUids =
            separateComponents ? null : new HashSet<String>()
    if (includeFare) {
        BigDecimal fare =
                MiscUtil.guarded(adapter.getEquivalentFare(prod, passengerType) as BigDecimal)
        if (separateComponents) {
            VatComponent comp = new VatComponent()
            comp.setBasis(fare)
            comp.setRate(Double.valueOf(rate))
            comp.setSum(calculateVatAmount(fare, rate))
            comp.getBasisTypes().add(VatBasisType.FARE)
            result.add(comp)
        } else {
            totalBasis = totalBasis.add(fare)
        }
    }
    Map<String, List<T>> taxesMap = new LinkedHashMap<>()
    for (T tax : adapter.getTaxes(prod, passengerType)) {
        String taxCode = adapter.getTaxCode(tax)
        if (taxesParams.excludeTaxesCodes && (taxCode!= null)
                && taxesParams.taxCodes.contains(taxCode)) {
            continue
        }
        if (!taxesParams.excludeTaxesCodes && ((taxCode == null)
                || !taxesParams.taxCodes.contains(taxCode))) {
            continue
        }
        String key = groupTaxesWithSameCodes && taxCode != null? taxCode: adapter.getUid(tax)
        List<T> taxes = taxesMap.get(key)
        if(taxes == null){
            taxes = new ArrayList<>()
            taxesMap.put(key, taxes)
        }
        taxes.add(tax)
    }
    for (List<T> taxes : taxesMap.values()) {
        BigDecimal basis = BigDecimal.ZERO
        Set<String> taxesUids2 = new HashSet<>()
        for(T tax: taxes){
            basis = MiscUtil.sum(basis, adapter.getEquivalentAmount(tax))
            taxesUids2.add(adapter.getUid(tax))
        }
        if (separateComponents) {
            VatComponent comp = new VatComponent()
            comp.setBasis(basis)
            comp.setRate(Double.valueOf(rate))
            comp.setSum(calculateVatAmount(basis, rate))
            comp.getBasisTypes().add(VatBasisType.TAXES)
            comp.getTaxesUids().addAll(taxesUids2)
            result.add(comp)
        } else {
            totalBasis = totalBasis.add(basis)
            taxesUids.addAll(taxesUids2)
        }
    }
    if (!separateComponents && (!taxesUids.isEmpty() || includeFare)) {
        VatComponent comp = new VatComponent()
        comp.setBasis(totalBasis)
        comp.setRate(Double.valueOf(rate))
        comp.setSum(calculateVatAmount(totalBasis, rate))
        if (includeFare) {
            comp.getBasisTypes().add(VatBasisType.FARE)
        }
        if (!taxesUids.isEmpty()) {
            comp.getBasisTypes().add(VatBasisType.TAXES)
            comp.getTaxesUids().addAll(taxesUids)
        }
        result.add(comp)
    }
    return result
}
private static<P, T, ST,S> boolean isInternal(final P product,
                                             final String countryCode, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    if (TextUtil.isBlank(countryCode)) {
        return false
    }
    for (ST st : adapter.getSegmentTariffs(product)) {
        for (S seg : adapter.getSegments(st)) {
            if (!AirProductHelper.isLocationInCountry(adapter.getDepartureLocation(seg),
                    countryCode)
                    || !AirProductHelper.isLocationInCountry(adapter.getArrivalLocation(seg),
                    countryCode)) {
                return false
            }
        }
    }
    return true
}

private static<P,T,ST,S> TransportationType getTransportationType(final P prod, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {

    boolean hasDomestic = false
    boolean hasInternational = false
    boolean hasZeroVVL = false
    boolean hasZVL = false
    boolean hasMoscow = false
    for (ST st : adapter.getSegmentTariffs(prod)) {
        for (S seg : adapter.getSegments(st)) {
            DictionaryReference<GeoLocation> arriveLocation =
                    adapter.getArrivalLocation(seg)
            DictionaryReference<GeoLocation> departureLocation =
                    adapter.getDepartureLocation(seg)
            String arrive =
                    arriveLocation != null ? arriveLocation.getCode() : null
            String depart =
                    departureLocation != null ? departureLocation.getCode()
                            : null
            Country arriveCountry = DictHelper.findCountry(arrive)
            Country departCountry = DictHelper.findCountry(depart)

            if ((arriveCountry == null) || (departCountry == null)) {
                continue
            }

            boolean departureIsDomestic = departCountry.isDomestic()
            boolean departureIsZeroVVL =
                    isCrimea(departureLocation) ||isKaliningrad(departureLocation) || isDVFO(departureLocation)
            boolean departureIsMoscow = AirProductHelper.isMoscow(departureLocation)
            boolean arrivalIsDomestic = arriveCountry.isDomestic()
            boolean arrivalIsZeroVVL =
                    isCrimea(arriveLocation) || isKaliningrad(arriveLocation) || isDVFO(arriveLocation)
            boolean arrivalIsMoscow = AirProductHelper.isMoscow(arriveLocation)
            if (!departureIsDomestic && !arrivalIsDomestic) {
                hasZVL = true
            } else if (!departureIsDomestic || !arrivalIsDomestic) {
                hasInternational = true
            } else if (departureIsZeroVVL || arrivalIsZeroVVL) {
                hasZeroVVL = true
            } else {
                hasDomestic = true
            }
            if(arrivalIsMoscow || departureIsMoscow){
                hasMoscow = true
            }
        }
    }
    if (!hasZVL && !hasDomestic && !hasInternational && !hasZeroVVL
            && !hasZVL&&!hasMoscow) {
        return TransportationType.UNDEFINED
    }
    if (hasZVL && !hasDomestic && !hasInternational && !hasZeroVVL) {
        return TransportationType.ZVL
    }
    if (hasInternational) {
        return TransportationType.MVL
    }
    if (hasZeroVVL) {
        return TransportationType.ZERO_VVL
    }
    if(!hasMoscow){
        return TransportationType.ZERO_VVL
    }
    return TransportationType.VVL
}

private static boolean isCrimea(final DictionaryReference<GeoLocation> loc) {
    if (loc == null) {
        return false
    }

    return "SIP" == loc.getCode() || "airport SIP" == loc.getCode() || "Севастополь" == loc.getCode() || "UKS" == loc.getCode()|| "airport UKS" == loc.getCode()
}

private static boolean isKaliningrad(
        final DictionaryReference<GeoLocation> loc) {
    if (loc == null) {
        return false
    }

    return "KGD" == loc.getCode() || "airport KGD" == loc.getCode()
}

private static boolean isDVFO(
        final DictionaryReference<GeoLocation> locRef) {
    return isDVFO(locRef, new HashSet<>())
}

private static boolean isDVFO(final DictionaryReference<GeoLocation> locRef,
                              final Set<DictionaryReference<GeoLocation>> processed) {
    GeoLocation loc = DictionaryCache.get().resolveReference(locRef)
    if (loc == null) {
        return false
    }
    if (loc.getRegions().contains(dvfoRegion)) {
        return true
    }
    if (loc.getParent() == null) {
        return false
    }
    if (processed.contains(loc.getParent())) {
        return false
    }
    processed.add(locRef)
    return isDVFO(loc.getParent(), processed)
}

private static<P,T,ST,S> boolean isHasVat(final P product, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {

    TransportationType transportationType = getTransportationType(adapter.getTransportationTypeProduct(product), adapter)
    return transportationType == TransportationType.VVL

}

private static Date getDate(int year, int month, int day) {
    Calendar cal = Calendar.getInstance()
    cal.set(Calendar.YEAR, year)
    cal.set(Calendar.MONTH, month)
    cal.set(Calendar.DAY_OF_MONTH, day)
    cal.set(Calendar.HOUR_OF_DAY, 0)
    cal.set(Calendar.MINUTE, 0)
    cal.set(Calendar.SECOND, 0)
    cal.set(Calendar.MILLISECOND, 0)
    return cal.getTime()
}

private static<P,T,ST,S> double getReducedVATRate(final P prod, AirProductVatCalculator.VatProductAdapter<P,T,ST,S> adapter) {
    final BigDecimal vat = DictHelper.getDefaultVat()
    double defaultVat = vat != null ? vat.doubleValue() : 18d
    //XTR-3715
    if (SystemHelper.isRussianInstallation()) {
        Date date = adapter.getVatIssueDate(prod)
        if (date != null) {
            Calendar cal = Calendar.getInstance()
            cal.setTime(date)
            int year = cal.get(Calendar.YEAR)
            int month = cal.get(Calendar.MONTH)
            if ((year > 2015)
                    || ((year == 2015) && (month >= Calendar.JULY))) {
                return 10
            }
        }
        return 10
    }
    return defaultVat
}

/*constants*/
@Field final static TaxesVatCalculationParams zzTaxComponent = new TaxesVatCalculationParams(Collections.singleton("ZZ"), false)

@Field final static TaxesVatCalculationParams taxesExceptZZ= new TaxesVatCalculationParams(Arrays.asList("ZZ", "CP"), true)

@Field final static TaxesVatCalculationParams kzTaxComponent = new TaxesVatCalculationParams(Collections.emptyList(), true)

@Field final  static String KZ_COUNTRY_CODE = "KZ"

@Field final  static  TaxesVatCalculationParams taxesExceptZZ_RI_YQ_YR_TR_M6_CU_Q5 =
        new TaxesVatCalculationParams(Arrays.asList("ZZ", "RI","YQ","YR", "TR", "M6", "CU", "Q5"), true)

@Field final  static  TaxesVatCalculationParams yrTaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("YR"), false)

@Field final  static  TaxesVatCalculationParams yqTaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("YQ"), false)

@Field final  static  TaxesVatCalculationParams riTaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("RI"), false)

@Field final  static  TaxesVatCalculationParams trTaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("TR"), false)

@Field final  static  TaxesVatCalculationParams m6TaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("M6"), false)

@Field final  static  TaxesVatCalculationParams cuTaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("CU"), false)

@Field final  static  TaxesVatCalculationParams q5TaxCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("Q5"), false)

@Field final static DictionaryReference<GeoRegion> dvfoRegion =
        new GeoRegionReference("DVFO")

@Field static final TaxesVatCalculationParams fareComponent =
        new TaxesVatCalculationParams(Collections.<String> emptyList(), false)

@Field static final TaxesVatCalculationParams suFuelChargeComponent = new TaxesVatCalculationParams(
        Arrays.asList(AirProductTaxHelper.getFuelTaxesCodes(new AirlineReference("SU"))), false)

@Field static final List<TaxesVatCalculationParams> tchReduceVATTaxesComps =
        Arrays.asList(
                new TaxesVatCalculationParams(
                        Collections.singleton("YQ"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("RU"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("PS"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("SA"), false))
@Field static final List<TaxesVatCalculationParams> tchReduceVATTaxesWithRIComps =
        Arrays.asList(
                new TaxesVatCalculationParams(
                        Collections.singleton("YQ"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("RU"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("PS"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("SA"), false),
                new TaxesVatCalculationParams(
                        Collections.singleton("RI"), false))

//XTR-6501
@Field static final List<String> noRICarriers = Arrays.asList("SU","DP", "UT")

@Field  static final TaxesVatCalculationParams utTaxComponent =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "YR", "SA", "PS"),
                false)

@Field static final TaxesVatCalculationParams s7TaxComponents =
        new TaxesVatCalculationParams(Arrays.asList("YR", "YQ", "RI"), false)

@Field static final TaxesVatCalculationParams dpStandardTaxesVatCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "SA", "PS", "RU"), false)
@Field static final TaxesVatCalculationParams dpB110TaxesCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("B110", "B", "B10", "B20"),
                false)
@Field static final TaxesVatCalculationParams dpSMSTaxesCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList("SMSC", "SMSD", "SMSF", "SMSP"),
                false)
@Field static final TaxesVatCalculationParams dpRestTaxesVatCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "YR", "SA", "PS", "RU", "ZZ", "B110", "B", "B10", "B20",
                "CP", "SMSC", "SMSD", "SMSF", "SMSP"), true)

@Field static final TaxesVatCalculationParams defaultStandardTaxesVatCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "YR", "SA", "PS", "RU"), false)
@Field static final TaxesVatCalculationParams defaultStandardTaxesWithRIVatCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "YR", "SA", "PS", "RU", "RI"), false)

@Field static final TaxesVatCalculationParams defaultRestTaxesVatCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "YR", "SA", "PS", "RU","ZZ","CP"), true)
@Field static final TaxesVatCalculationParams defaultRestTaxesWithoutRIVatCalculationParams =
        new TaxesVatCalculationParams(Arrays.asList(
                "YQ", "YR", "SA", "PS", "RU", "ZZ", "RI","CP"), true)

@Field static final Date date_2020_10_01 = getDate(2020,Calendar.OCTOBER, 1)

/* main function*/

updateVatDetalization(detalizationProperty, productProperty, passengerTypeProperty, adapterProperty)